package dits_def;

# 	$Id: dits_def.pm,v 2.8 2001/08/14 14:07:26 p004184 Exp $	

# -----------------------------------------------------------------------
#
# Description:
#
#	Dfinitions des commandes systme utilises dans les templates
#	ainsi que de quelques fonctions utilitaires pour ITO.
#
#	L'optique est de fournir le plus possible de commandes avec une
#	implmentation en perl.
# 	Pour les autres commandes unix, cration d'une fonction appelant
#	la commande systeme et ayant le mme nom (et si possible les
#	mmes possibilits).
#
#	Un programme utilisant le module peut utiliser directement les
#	commandes fournies, la syntaxe pouvant varier de la syntaxe des
#	vraies commandes UNIX :
#		ps("-ef");
#		cat("toto","tata","titi");
#
# usage:
#
#	use dits_def;
#	@result = df();
#	Trace "Erreur";
#
# Note sur SelfLoader :
#
# L'utilisation de SelfLoader permet de ne charger que les fonctions
# rellement utilises. Un gain immdiat est 100Ko en moins en mmoire.
# Quelques time montrent peu de diffrences pour la CPU : gain de 0.020s
# pour la CPU user et de 0.010s pour la CPU sys. Au total le cot de
# l'utilisation de SelfLoader est compltement amorti et on peut donc
# le conserver sans remord.
# Un script sh prend au moins 100Ko. Un script perl utilisant cette
# librairie prend aux alentours de 450Ko.
#
# -----------------------------------------------------------------------
#
# AJOUT DE COMMANDES
#
#	Ajouter si ncessaire une commande UNIX dans la table %CMD
#	pour les diffrents OS supports.
#	Implmenter la fonction en s'inspirant des fonctions existantes.
#	Ajouter le nom de la fonction dans @EXPORT.
#
# FONCTIONS FOURNIES (liste non exhaustive)
#
#	bdf [options]
#	cat [file...]
#	cfgtest <user> [options]
#	cftutil <user> <repertoire> <commande>
#	date [format]
#	df [options]
#	dspstat <user> <host a tester> [options]
#   ifconfig [options]
#	ipcrm [options]
#	ipcs [options]
#	haget [options]
#	llbd [options]
#	mail <message> [options]
#	metastat [options]
#	metaset [options]
#	opcmon [options]
#	opcmsg <application> <object> <msg_text> <severity> <msg_grp> [node]
#		nota : il vaut mieux ne pas forcer le node et laisser ITO travailler
#	opcagt [options]
#	opcackmsg [options]
#	opcctla [options]
#	opcctlm [options]
#	opcragt [options]
#	opcsv [options]
#	ovstart [options]
#	ovstop [options]
#	powermt [options]
#	ps [options]
#	rsh [options]
#	sqlplus <requete>
#	sendmail [options]
#	su <user> <commande>
#	tail [options]
#	top [options]
#	ypcat [options]
#	vxprint [options]
#	vxdisk [options]
#	vxdctl [options]
#	vxiod [options]
#
#	VerifCommande <commande...>
#		vrifie que la commande existe pour l'OS
#
#	Trace <texte>
#		affiche le texte sur STDOUT et dans un fichier log
#
#	CftDir
#		renvoie les rpertoires CFT sur la machine
#
#	CftLogs
#		renvoie les fichiers de log CFT sur la machine
#
#	CftRunning <repertoire>
#		indique si cft tourne ou non
#
#	InstanceOracle <hostname>
#		renvoie une liste des instances Oracle trouves
#		sur la machine
#
#	CronLog
#		renvoie le nom du fichier log de cron sur la machine
#
#	OpcLog
#		renvoie le rpertoire des logs OpC sur la machine
#
#	OpcTmp
#		renvoie le rpertoire des fichiers temporaires OpC
#
#	SysLog
#		renvoie le nom du fichier syslog sur la machine
#
#	Debug
#		Renvoie 1 si le fichier /var/tmp/opc_debug est prsent
#		sinon renvoie 0
#		Les scripts utilisant cette fonction pourront tre
#		facilement dbuggs avec un 'touch /var/tmp/opc_debug'
#
#	StatGdg <hostname> <instance gdg>
#		Renvoie le fichier des statistiques GDG du jour
#
#	ProcessPresent <hostname> <user> <pid>
#		Vrifie que le process spcifi existe
#
#	IsCluster
#		Indique si le host fait partie d'un cluster
#
#	FlexVersion
#		Renvoie les versions de Flex prsentes sur la machine
#
#	FlexServeur
#		Indique si la machine est serveur Flex
#
#	FlexStat <version> [options]
#		Utilitaire Flex pour le status des licences
#
#	FlexServers <version> <licence>
#		Renvoie les serveurs de la licence
#
#	FlexFile <version> <licence>
#		Renvoie le nom du fichier de paramtrage de la licence
#
#	FlexDaemon <version> <licence>
#		Renvoie le dmon de la licence ainsi que sa localisation
#
#	FlexFeatures <version> <licence>
#		Renvoie les features de la licence
#
# -----------------------------------------------------------------------

require 5.003;
use strict;
use Fcntl ':flock';
use File::Basename;
use POSIX 'strftime';

use SelfLoader;
require Exporter;
use vars qw(@ISA @EXPORT $VERSION %CMD $OS);
@ISA = qw(Exporter);
$VERSION	 = "1.0";

@EXPORT = qw(at bdf cat cfgtest cftutil CftDir CftLogs date df dspstat haget ifconfig ipcrm ipcs ipssend
			 ncs mail metaset metastat metadb netstat quota iostat
			 opcmon opcmsg opcagt opcsv ovstart ovstop ovstatus logger
			 oralsn ovoracle scconf NomCluster hastat pnmstat hareg haoracle
			 opcackmsg opcctla opcctlm opcragt powermt ps rsh sqlplus sendmail
			 su tail top ypcat ypmatch ypwhich du nslookup
			 vxprint vxdisk vxdctl vxiod vxdg servOCSEPS servlicenceOCSEPS
			 ping objcall wping wlookup domainname
			 VerifCommande Trace CftRunning InstanceOracle
			 FlexVersion FlexStat FlexServeur FlexServers FlexDaemon FlexFeatures
			 FlexFile FlexPorts exportfs NFS_process
			 CronLog SysLog Debug StatGdg OpcLog OpcTmp IsCluster NomCluster NFSServeur
			 pfiles pstack pmap sar NbProc xlvmgr);

$CMD{Solaris}{AT}		= "/usr/bin/at";
$CMD{Solaris}{BDF}		= "/usr/bin/df -lk";
$CMD{Solaris}{CRONLOG}	= "/var/cron/log";
$CMD{Solaris}{DEBUG}	= "/var/tmp/opc_debug";
$CMD{Solaris}{DF}		= "/usr/bin/df";
$CMD{Solaris}{DOMAINNAME} = "/usr/bin/domainname";
$CMD{Solaris}{DU}		= "/usr/bin/du";
$CMD{Solaris}{EXPORTFS}	= "/usr/sbin/exportfs";
$CMD{Solaris}{HAGET}	= "/opt/SUNWcluster/bin/haget";
$CMD{Solaris}{HAGET2}	= "/opt/SUNWhadf/bin/haget";
$CMD{Solaris}{HAREG}	= "/opt/SUNWcluster/bin/hareg";
$CMD{Solaris}{HAREG2}	= "/opt/SUNWhadf/bin/hareg";
$CMD{Solaris}{HASTAT}	= "/opt/SUNWcluster/bin/hastat";
$CMD{Solaris}{HASTAT2}	= "/opt/SUNWhadf/bin/hastat";
$CMD{Solaris}{HAORACLE}	= "/opt/SUNWcluster/bin/haoracle";
$CMD{Solaris}{HAORACLE2} = "/opt/SUNWhadf/bin/haoracle";
$CMD{Solaris}{IFCONFIG} = "/usr/sbin/ifconfig";
$CMD{Solaris}{INITD}	= "/etc/init.d";
$CMD{Solaris}{IOSTAT}	= "/usr/bin/iostat";
$CMD{Solaris}{IPCRM}	= "/usr/bin/ipcrm";
$CMD{Solaris}{IPCS}		= "/usr/bin/ipcs";
$CMD{Solaris}{LOGGER}	= "/usr/bin/logger";
$CMD{Solaris}{METADB}	= "/usr/opt/SUNWmd/sbin/metadb";
$CMD{Solaris}{METASET}	= "/usr/opt/SUNWmd/sbin/metaset";
$CMD{Solaris}{METASTAT} = "/usr/opt/SUNWmd/sbin/metastat";
$CMD{Solaris}{MAIL}		= "/usr/bin/mailx";
$CMD{Solaris}{NCS}		= "/etc/init.d/ncs";
$CMD{Solaris}{NETSTAT}	= "/usr/bin/netstat";
$CMD{Solaris}{NSLOOKUP}	= "/usr/sbin/nslookup";
$CMD{Solaris}{NOMCLUSTER} = "/etc/opt/SUNWcluster/conf/default_clustername";
$CMD{Solaris}{OPCLOG}	= "/var/opt/OV/log/OpC";
$CMD{Solaris}{OPCTMP}	= "/var/opt/OV/tmp/OpC";
$CMD{Solaris}{OPCMON}	= "/opt/OV/bin/OpC/opcmon";
$CMD{Solaris}{OPCMSG}	= "/opt/OV/bin/OpC/opcmsg";
$CMD{Solaris}{OPCAGT}	= "/opt/OV/bin/OpC/opcagt";
$CMD{Solaris}{OPCACKMSG}= "/opt/OV/bin/OpC/opcackmsg";
$CMD{Solaris}{OPCCTLA}	= "/opt/OV/bin/OpC/opcctla";
$CMD{Solaris}{OPCRAGT}	= "/opt/OV/bin/OpC/opcragt";
$CMD{Solaris}{PFILES}	= "/usr/proc/bin/pfiles";
$CMD{Solaris}{PSTACK}	= "/usr/proc/bin/pstack";
$CMD{Solaris}{PMAP}		= "/usr/proc/bin/pmap";
$CMD{Solaris}{PING}		= "/usr/sbin/ping";
$CMD{Solaris}{PNMSTAT}	= "/opt/SUNWpnm/bin/pnmstat";
$CMD{Solaris}{POWERMT}	= "/etc/powermt";
$CMD{Solaris}{PS}		= "/usr/bin/ps";
$CMD{Solaris}{QUOTA}	= "/usr/sbin/quota";
$CMD{Solaris}{RSH}		= "/usr/bin/rsh";
$CMD{Solaris}{SCCONF}	= "/opt/SUNWcluster/bin/scconf";
$CMD{Solaris}{SENDMAIL}	= "/usr/lib/sendmail";
$CMD{Solaris}{SERVOCSEPS} = "/etc/init.d/servOCSEPS";
$CMD{Solaris}{SERVLICENCEOCSEPS} = "/etc/init.d/servlicenceOCSEPS";
$CMD{Solaris}{SAR}		= "/usr/sbin/sar";
$CMD{Solaris}{SHARE}	= "/usr/sbin/share";
$CMD{Solaris}{SU}		= "/sbin/su";
$CMD{Solaris}{SYSLOG}	= "/var/adm/messages";
$CMD{Solaris}{TAIL}		= "/bin/tail";
$CMD{Solaris}{TIVOLI}	= "/etc/Tivoli/setup_env.sh";
$CMD{Solaris}{TOP}		= "/usr/bin/ps -e -o tty,pid,user,pri,nice,rss,vsz,s,time,pcpu,pcpu,args";
$CMD{Solaris}{VXPRINT}	= "/usr/sbin/vxprint";
$CMD{Solaris}{VXDISK}	= "/usr/sbin/vxdisk";
$CMD{Solaris}{VXDCTL}	= "/usr/sbin/vxdctl";
$CMD{Solaris}{VXIOD}	= "/usr/sbin/vxiod";
$CMD{Solaris}{VXDG}		= "/usr/sbin/vxdg";
$CMD{Solaris}{YPCAT}	= "/usr/bin/ypcat";
$CMD{Solaris}{YPMATCH}	= "/usr/bin/ypmatch";
$CMD{Solaris}{YPWHICH}	= "/usr/bin/ypwhich";

$CMD{HPUX10}{BDF}		= "/usr/bin/bdf -l";
$CMD{HPUX10}{CRONLOG}	= "/var/adm/cron/log";
$CMD{HPUX10}{DEBUG}		= "/var/tmp/opc_debug";
$CMD{HPUX10}{DF}		= "/usr/bin/df";
$CMD{HPUX10}{DOMAINNAME} = "/usr/bin/domainname";
$CMD{HPUX10}{DU}		= "/usr/bin/du";
$CMD{HPUX10}{EXPORTFS}	= "/usr/sbin/exportfs";
#$CMD{HPUX10}{IFCONFIG}  = "/usr/sbin/ifconfig";
$CMD{HPUX10}{INITD}		= "/sbin/init.d";
$CMD{HPUX10}{IPCRM}		= "/usr/bin/ipcrm";
$CMD{HPUX10}{IPCS}		= "/usr/bin/ipcs";
$CMD{HPUX10}{LOGGER}	= "/usr/bin/logger";
$CMD{HPUX10}{MAIL}		= "/usr/bin/mailx";
$CMD{HPUX10}{NETSTAT}	= "/usr/bin/netstat";
$CMD{HPUX10}{NSLOOKUP}	= "/usr/bin/nslookup";
$CMD{HPUX10}{OPCLOG}	= "/var/opt/OV/log/OpC";
$CMD{HPUX10}{OPCTMP}	= "/var/opt/OV/tmp/OpC";
$CMD{HPUX10}{OPCMON}	= "/opt/OV/bin/OpC/opcmon";
$CMD{HPUX10}{OPCMSG}	= "/opt/OV/bin/OpC/opcmsg";
$CMD{HPUX10}{OPCAGT}	= "/opt/OV/bin/OpC/opcagt";
$CMD{HPUX10}{OPCSV}		= "/opt/OV/bin/OpC/opcsv";
$CMD{HPUX10}{OPCACKMSG}	= "/opt/OV/bin/OpC/opcackmsg";
$CMD{HPUX10}{OPCCTLA}	= "/opt/OV/bin/OpC/opcctla";
$CMD{HPUX10}{OPCCTLM}	= "/opt/OV/bin/OpC/opcctlm";
$CMD{HPUX10}{OVSTOP}	= "/opt/OV/bin/ovstop";
$CMD{HPUX10}{OVSTART}	= "/opt/OV/bin/ovstart";
$CMD{HPUX10}{OVSTATUS}	= "/opt/OV/bin/ovstatus";
$CMD{HPUX10}{ORALSN}	= "/sbin/init.d/oralsn";
$CMD{HPUX10}{OVORACLE}	= "/sbin/init.d/ovoracle";
$CMD{HPUX10}{OPCRAGT}	= "/opt/OV/bin/OpC/opcragt";
$CMD{HPUX10}{PING}		= "/usr/sbin/ping";
$CMD{HPUX10}{PS}		= "/bin/ps";
$CMD{HPUX10}{QUOTA}		= "/usr/bin/quota";
$CMD{HPUX10}{RSH}		= "/usr/bin/rsh";
$CMD{HPUX10}{SAR}		= "/usr/sbin/sar";
$CMD{HPUX10}{SENDMAIL}	= "/usr/sbin/sendmail";
$CMD{HPUX10}{SU}		= "/usr/bin/su";
$CMD{HPUX10}{SYSLOG}	= "/var/adm/syslog/syslog.log";
$CMD{HPUX10}{TAIL}		= "/bin/tail";
#$CMD{HPUX10}{TOP}		= "/usr/bin/top -d 1";	# top renvoi des codes de controle a filtrer
$CMD{HPUX10}{YPCAT}		= "/usr/bin/ypcat";
$CMD{HPUX10}{YPMATCH}	= "/usr/bin/ypmatch";
$CMD{HPUX10}{YPWHICH}	= "/usr/bin/ypwhich";

$CMD{IRIX}{BDF}			= "/usr/bin/df -lk";
$CMD{IRIX}{CHKCONFIG}	= "/sbin/chkconfig";
$CMD{IRIX}{CRONLOG}		= "/var/cron/log";
$CMD{IRIX}{DEBUG}		= "/var/tmp/opc_debug";
$CMD{IRIX}{DF}			= "/usr/bin/df";
$CMD{IRIX}{DOMAINNAME}	= "/bin/domainname";
$CMD{IRIX}{DU}			= "/usr/bin/du";
$CMD{IRIX}{EXPORTFS}	= "/usr/etc/exportfs";
$CMD{IRIX}{IFCONFIG}    = "/usr/etc/ifconfig";
$CMD{IRIX}{INITD}		= "/etc/init.d";
$CMD{IRIX}{IPCRM}		= "/usr/bin/ipcrm";
$CMD{IRIX}{IPCS}		= "/usr/bin/ipcs";
$CMD{IRIX}{LOGGER}		= "/usr/bsd/logger";
$CMD{IRIX}{MAIL}		= "/usr/sbin/Mail";
$CMD{IRIX}{NETSTAT}		= "/usr/etc/netstat";
$CMD{IRIX}{NSLOOKUP}	= "/usr/sbin/nslookup";
$CMD{IRIX}{OPCLOG}		= "/var/opt/OV/log/OpC";
$CMD{IRIX}{OPCTMP}		= "/var/opt/OV/tmp/OpC";
$CMD{IRIX}{OPCMON}		= "/opt/OV/bin/OpC/opcmon";
$CMD{IRIX}{OPCMSG}		= "/opt/OV/bin/OpC/opcmsg";
$CMD{IRIX}{OPCAGT}		= "/opt/OV/bin/OpC/opcagt";
$CMD{IRIX}{OPCACKMSG}	= "/opt/OV/bin/OpC/opcackmsg";
$CMD{IRIX}{OPCCTLA}		= "/opt/OV/bin/OpC/opcctla";
$CMD{IRIX}{OPCRAGT}		= "/opt/OV/bin/OpC/opcragt";
$CMD{IRIX}{PING}		= "/usr/etc/ping";
$CMD{IRIX}{PS}			= "/bin/ps";
$CMD{IRIX}{QUOTA}		= "/usr/bsd/quota";
$CMD{IRIX}{RSH}			= "/usr/bsd/rsh";
$CMD{IRIX}{SAR}			= "/bin/sar";
$CMD{IRIX}{SENDMAIL}	= "/usr/lib/sendmail";
$CMD{IRIX}{SU}			= "/sbin/su";
$CMD{IRIX}{SYSLOG}		= "/var/adm/SYSLOG";
$CMD{IRIX}{TAIL}		= "/bin/tail";
$CMD{IRIX}{XLVMGR}      = "/sbin/xlv_mgr";
$CMD{IRIX}{YPCAT}		= "/usr/bin/ypcat";
$CMD{IRIX}{YPMATCH}		= "/usr/bin/ypmatch";
$CMD{IRIX}{YPWHICH}		= "/usr/bin/ypwhich";

#
# Protger les \ par des quotes simples...
#
$CMD{NT}{DEBUG}			= '\TEMP\debug.opc';
$CMD{NT}{OPCLOG}		= '\usr\OV\log\OpC';
$CMD{NT}{OPCMON}		= '\usr\OV\bin\OpC\intel\opcmon.exe';
$CMD{NT}{OPCMSG}		= '\usr\OV\bin\OpC\intel\opcmsg.exe';
$CMD{NT}{PS}			= '\usr\OV\bin\OpC\intel\cmds\itodiag.exe /processes';
$CMD{NT}{IPSSEND}       = '\win32app\raidinfo\ipssend.exe';

# --------------------------------------------------------
# fonctions utilitaires non exportees
# --------------------------------------------------------

sub timeout {
	my ($param,$delai) = @_;
	my(@result);
	eval {
		local $SIG{ALRM} = sub { die "timeout\n" };
		alarm $delai;
		@result = `$param`;
		alarm 0;
	};
	@result = $@ if($@);
	sleep 1;
	return @result;
}

# fonction appele par les fonctions standard
sub standard {
	my ($commande,@param) = @_;
	my @result = "Commande non supportee\n";
	my $code = 1;
	if($CMD{$OS}{$commande}) {
		@result = `$CMD{$OS}{$commande} @param`;
		$code = $?;
	}
	return ($code,@result);
}

# --------------------------------------------------------
# fonctions toujours utilises donc toujours exportees
# --------------------------------------------------------

sub Debug {
	return ($CMD{$OS}{DEBUG} && -f "$CMD{$OS}{DEBUG}");
}

sub Trace {
	my @param = @_;
	my $texte = date("%d-%m-%Y %H:%M:%S") . " [" . basename($0) . "] @param\n";
	print $texte;
	my $file = "$CMD{$OS}{OPCLOG}/opc.log";
	$file = "$CMD{$OS}{OPCLOG}\opc.log" if($CMD{$OS}{OPCLOG} =~ /^\\/);
	if(open(LOG,">>$file") && flock(LOG,LOCK_EX)) {
		print LOG $texte;
		flock(LOG,LOCK_UN);
		close(LOG);
	}
}

sub VerifCommande {
	my $ok=1;
	for(@_) {
		if(! $CMD{$OS}{$_}) {
			$ok=0;
		} else {
			my($commande) = split(/\s+/,$CMD{$OS}{$_});
			if(! -x $commande) {
				Trace "Executable $commande non trouve sur la machine" if(Debug());
				$ok=0;
			}
		}
	}
	return $ok;
}

sub opcmon {
	my @param = @_;
	my @result = "Commande non supportee";
	my $code = 1;
	if(VerifCommande("OPCMON")) {
		@result = `$CMD{$OS}{OPCMON} @param`;
		$code = $?;
		print "@result";
		if($code != 0) {
			my $texte = "Erreur opcmon code = " . ($code >> 8) . "-" . ($code & 255) . " : $!";
			opcmsg("ITO","opcmon","$texte","Major","OpC");
		}
	} else {
		opcmsg("ITO","opcmon","opcmon @param","Major","OpC");
	}
	return $code;
}

# --------------------------------------------------------
# determination de l'OS et des commandes systeme disponibles
# --------------------------------------------------------
sub DeterminationOS {
	if(%ENV && $ENV{'OS'} && $ENV{'OS'} eq 'Windows_NT') {
		$OS="NT";
	} else {
		chomp(my($type,$host,$version)=split(/\s+/,`uname -a`));
		if($type =~ /SunOS/) {
			$OS="Solaris";
			my ($major) = split(/\./,$version);
			$OS="SunOS" if($major < 5);
		} elsif($type =~ /HP-UX/) {
			$OS="HPUX";
			$OS="HPUX10" if($version =~ /B\.10/);
		} elsif($type =~ /IRIX/) {
			$OS="IRIX";
		} else {
			Trace "$type $version sur $host non supporte par ce programme";
			return 0;
		}
	}
	return 1;
}

DeterminationOS();

1;

__DATA__

# --------------------------------------------------------
# Aprs ce tag __DATA__ se trouvent les fonctions
# compiles seulement si elles sont utilises
# --------------------------------------------------------

# --------------------------------------------------------
# fonctions utilitaires exportees
# --------------------------------------------------------

sub OpcLog {
	return $CMD{$OS}{OPCLOG} if(-d $CMD{$OS}{OPCLOG});
	if($OS eq "NT") {
		return "\TEMP" if(-d "\TEMP");
	} else {
		return "/var/tmp" if(-d "/var/tmp");
		return "/tmp" if(-d "/tmp");
	}
	return " ";
}

sub OpcTmp {
	return $CMD{$OS}{OPCTMP} if(-d $CMD{$OS}{OPCTMP});
	return " ";
}

sub CronLog {
	return "" if(! $CMD{$OS}{CRONLOG});
	return $CMD{$OS}{CRONLOG};
}

sub SysLog {
	return "" if(! $CMD{$OS}{SYSLOG});
	return $CMD{$OS}{SYSLOG};
}

sub StatGdg {
	my($hostname,$user) = @_;
	$hostname =~ s/^phys-//;
	my $date = POSIX::strftime('%y%m%d',localtime());
	my $log = "/logiciel/$user/STATGDG/DATA/dspstat/dspstat.$date";
	$log = "$hostname/$log" if(! -f $log);
	$log = "" if(! -f $log);
	return $log;
}

sub ProcessPresent {
	my($hostname,$user,$pid) = @_;
	my @result;
	if($hostname) {
		@result = rsh("$hostname $CMD{$OS}{PS} -ef");
	} else {
		@result = ps("-ef");
	}
	for(@result) {
		next if(/PID/);
		my($proc_user,$proc_pid) = split(/\s+/);
		return 1 if((! $user || $proc_user eq $user)
					&& (!$pid || $proc_pid == $pid));
	}
	return 0;
}

sub IsCluster {
	return ($CMD{$OS}{NOMCLUSTER} && -f $CMD{$OS}{NOMCLUSTER});
}

sub NomCluster {
	my $cluster = "";
	if($CMD{$OS}{NOMCLUSTER} && -f $CMD{$OS}{NOMCLUSTER}
	   && open(FIC,$CMD{$OS}{NOMCLUSTER})) {
		chomp($cluster = <FIC>);
		close(FIC);
	}
	return $cluster;
}

sub FlexVersion {
	my(%version,@version,$file);
	if(opendir(DIR,"/serveur/Flex/licenses")) {
		while(defined($file = readdir(DIR))) {
			$version{$1} = $1 if($file =~ /^(\d+\.\w+)$/);
			$version{$1} = $1 if($file =~ /^licenses_(\d+\.\w+)$/);
		}
		closedir(DIR);
	}
	if(opendir(DIR,"/home/serveur/Flex/licenses")) {
		while(defined($file = readdir(DIR))) {
			$version{$1} = $1 if($file =~ /^(\d+\.\w+)$/);
			$version{$1} = $1 if($file =~ /^licenses_(\d+\.\w+)$/);
		}
		closedir(DIR);
	}
	for(sort keys %version) {
		push(@version,$_);
	}
	return @version;
}

sub FlexStat {
	my($version,@param) = @_;
	my @flex_stat;
	if(-f "/serveur/Flex/stat_$version") {
		@flex_stat = `/serveur/Flex/stat_$version @param 2> /dev/null`;
	} elsif(-f "/home/serveur/Flex/stat_$version") {
		@flex_stat = `/home/serveur/Flex/stat_$version @param 2> /dev/null`;
	} elsif(-f "/serveur/Flex/stat") {
		@flex_stat = `/serveur/Flex/stat $version @param 2> /dev/null`;
	} elsif(-f "/home/serveur/Flex/stat") {
		@flex_stat = `/home/serveur/Flex/stat $version @param 2> /dev/null`;
	} else {
		push(@flex_stat,"version $version non trouvee");
	}
	@flex_stat = ("version $version non trouvee")
		if(grep(/Actuellement ce script ne gere que/,@flex_stat)>0);
	@flex_stat = ("version $version non trouvee")
		if(@flex_stat - grep(/^\s*$/,@flex_stat) ==1);
	return @flex_stat;
}

sub FlexFile {
	my ($version,$licence) = @_;
	my @result = FlexStat($version,$licence);
	my $file = "/serveur/Flex/licenses/licenses_$version/license_${version}_$licence.dat";
	if(!-f $file) {
		$file = "/serveur/Flex/licenses/$version/${version}_$licence.dat";
	} elsif(!-f $file) {
		$file = "/home/serveur/Flex/licenses/licenses_$version/license_${version}_$licence.dat";
	} elsif(!-f $file) {
		$file = "/home/serveur/Flex/licenses/$version/${version}_$licence.dat";
	}
	for(@result) {
		if(/\(License file: (\S+\.dat)\)/) {
			$file = $1;
			last;
		} elsif(/License file\(s\) on \S+: (\S+\.dat)/) {
			$file = $1;
			last;
		}
	}
	return $file;
}

# lecture du fichier en prenant en compte les \ de continuation
# et en enlevant les commentaires
# on stocke le rsultat pour les demandes ultrieures
my (@flex_file,$flex_version,$flex_licence);
sub FlexLecture {
	my ($version,$licence) = @_;
	my $continuation = 0;
	my $ligne;
	if(!@flex_file || $version ne $flex_version || $licence ne $flex_licence) {
		$flex_version = $version;
		$flex_licence = $licence;
		undef @flex_file;
		my $file = FlexFile($version,$licence);
		if(-f "$file" && open(PARAM,"<$file")) {
			while(<PARAM>) {
				next if(/^\#/);
				if($continuation) {
					chomp($ligne = "$ligne $_");
					$continuation = 0;
				} else {
					chomp($ligne = $_);
				}
				if($ligne =~ /(.*)\\$/) {
					$ligne = $1;
					$continuation = 1;
					next;
				}
				push(@flex_file,$ligne);
			}
			close(PARAM);
		} else {
			Trace "$version/$licence: open $file : $!";
		}
	}
	return @flex_file;
}

sub FlexServeur {
	my @boot = ("/etc/init.d/servlicense","/etc/init.d/Flexlm","/etc/init.d/flexlm");
	for(@boot) {
		my $boot = $_;
		if(-f $boot && open(ETC,"<$boot")) {
			my $ligne = grep(!/^\s*\#/ && /FLEX|Flex/, <ETC>);
			close(ETC);
			return 1 if($ligne>0);
		}
	}
	return 0;
}

sub FlexServers {
	my ($version,$licence) = @_;
	my @serveurs;
	my @param = FlexLecture($version,$licence);
	for(@param) {
		push(@serveurs,$1) if(/^SERVER\s+(\S+)/);
	}
	return @serveurs;
}

sub FlexPorts {
	my ($version,$licence) = @_;
	my @ports;
	my @param = FlexLecture($version,$licence);
	for(@param) {
		push(@ports,$1) if(/^SERVER\s+\S+\s+\S+\s+(\d+)/);
	}
	return @ports;
}

sub FlexDaemon {
	my ($version,$licence) = @_;
	my @demon;
	my @param = FlexLecture($version,$licence);
	for(@param) {
		if(/^(DAEMON|VENDOR|vendor)\s+/) {
			my($rien,$daemon,$path) = split(/\s+/);
			@demon = ($daemon,$path);
			last;
		}
	}
	return @demon;
}

sub FlexFeatures {
	my ($version,$licence) = @_;
	my @features;
	my %compo;
	my $package = "n importe quoi";
	my @param = FlexLecture($version,$licence);
	for(@param) {
		if(/^PACKAGE\s+(\S+).*COMPONENTS=\"(.+)\"\s*$/) {
			$compo{$1} = $2;
		} elsif(/^(FEATURE|INCREMENT)\s+/) {
			my($rien,$feature,$daemon,$version,$date,$number) = split(/\s+/);
			if($compo{$feature}) {
				for(split(/\s+/,$compo{$feature})) {
					my($feature,$version,$num2) = split(/:/,$_);
					$num2 = $number if(!$num2);
					push(@features,[$feature,$date,$num2,$daemon]);
				}
			} else {
				push(@features,[$feature,$date,$number,$daemon]);
			}
		}
	}
	return @features;
}

sub NFSServeur {
	my $code = 0;
	if($CMD{$OS}{SHARE} && -f $CMD{$OS}{SHARE}) {
		my $tab = "/etc/dfs/dfstab";
		if(-f $tab && open(TAB,"<$tab")) {
			my $ligne = grep(/^share/, <TAB>);
			close(TAB);
			$code = 1 if($ligne>0);
		}
	} else {
		my $tab = "/etc/xtab";
		if(-f $tab && open(TAB,"<$tab")) {
			my $ligne = grep(!/^\#/, <TAB>);
			close(TAB);
			$code = 1 if($ligne>0);
		}
	}
	return $code;
}

# process ncessaires  NFS
# teste client/serveur et OS
sub NFS_process {
	my %process;
	if($OS eq 'Solaris') {
		my @fic;
		for("/etc/rc2.d","/etc/rc3.d") {
			if(opendir(DIR,$_)) {
				push(@fic,readdir(DIR));
				closedir(DIR);
			}
		}
		if(grep(/S\d+nfs\.client/,@fic)) {
			$process{'statd'}++;
			$process{'lockd'}++;
		}
		if(grep(/S\d+nfs\.server/,@fic)) {
			if(open(DFS,"</etc/dfs/dfstab")) {
				my @dfs = grep(!/^\#/ && /nfs/,<DFS>);
				close(DFS);
				if(@dfs) {
					if(open(SHARE,"</etc/dfs/sharetab")) {
						my @share = grep(!/^\#/ && /nfs/,<SHARE>);
						close(SHARE);
						if(@share) {
							$process{'nfsd'}++;
							$process{'mountd'}++;
						}
					}
				}
			}
		}
		if(grep(/S\d+pcnfs/,@fic)) {
			$process{'rpc.pcnfsd'}++;
		}
		if(grep(/S\d+autofs/,@fic)) {
			$process{'automountd'}++;
		}
	} elsif($OS eq 'IRIX') {
		my ($code,@result);
		$process{'inetd'}++;
		foreach my $daemon ('ipfilterd','gated','routed','mrouted','named','rsvpd','nsd',
							'timed','timeslave','rarpd','rwhod','snetd','cachefs') {
			($code,@result) = chkconfig($daemon);
			if($code==0) { # on
				$process{$daemon}++;
			}
		}
		if(-c '/dev/tcp') {
			$process{'rpcbind'}++;
			$process{'sac'}++;
		} else {
			$process{'portmap'}++;
		}
		($code,@result) = chkconfig('ypmaster');
		if($code==0) { # on
			$process{'rpc.passwd'}++;
		}
		if(-x '/etc/init.d/afs') {
			$process{'afs'}++;
		}
		($code,@result) = chkconfig('nfs');
		if($code==0) { # on
			$process{'biod'}++;
			$process{'nfsd'}++;
			($code,@result) = chkconfig('lockd');
			if($code==0) { # on
				$process{'rpc.statd'}++;
				$process{'rpc.lockd'}++ if(!$process{'nfsd'}); # voir man lockd
			}
			($code,@result) = chkconfig('autofs');
			if($code==0) { # on
				$process{'autofsd'}++;
			} else {
				($code,@result) = chkconfig('automount');
				if($code==0) { # on
					$process{'automount'}++;
				}
			}
			($code,@result) = chkconfig('pcnfsd');
			if($code==0) { # on
				$process{'rpc.pcnfsd'}++;
			}
		}
	} elsif($OS =~ /HPUX/) {
		if(open(CONF,"</etc/rc.config.d/nfsconf")) {
			my @conf = <CONF>;
			close(CONF);
			if(grep(/^NFS_CLIENT=1/,@conf)>0) {
				$process{'portmap'}++;
				$process{'rpc.statd'}++;
				$process{'biod'}++;
				$process{'rpc.lockd'}++;
			}
			if(grep(/^AUTOMOUNT=1/,@conf)>0) {
				$process{'automount|automountd'}++;
			}
			if(grep(/^NFS_SERVER=1/,@conf)>0) {
				if(grep(/^START_MOUNTD=1/,@conf)>0) {
					$process{'rpc.mountd'}++;
				}
				$process{'portmap'}++;
				$process{'nfsd'}++;
				$process{'rpc.statd'}++;
				$process{'rpc.lockd'}++;
			}
			if(grep(/^PCNFS_SERVER=1/,@conf)>0) {
				$process{'rpc.pcnfsd'}++;
			}
		}
	}
	my @sans_doublon = (sort keys %process);
	return @sans_doublon;
}

sub NbProc {
	my $nb_proc = 1;
	if(-x "/bin/hinv") { # IRIX
		my @result = `/bin/hinv`;
		for(@result) {
			next if(!/^\d+.+Processor\s*$/);
			($nb_proc) = split;
			last;
		}
	} elsif(-x "/usr/sbin/psrinfo") { # SUN
		my @result = `/usr/sbin/psrinfo`;
		$nb_proc = @result;
	} elsif(-x "/usr/sbin/dmesg") { # HP
		my @result = `/usr/sbin/dmesg`;
		$nb_proc = grep(/^\d+\s+processor\s*$/,@result);
	}
	return $nb_proc;
}

sub CftRunning {
	my ($rep) = @_;
	return (-f "$rep/LOCK"
			|| -f "$rep/filexe/LOCK"
			|| -f "$rep/v_cour/filexe/LOCK"
			|| -f "$rep/vcour/filexe/LOCK"
			|| -f "$rep/fillog/LOCK"
			|| -f "$rep/v_cour/fillog/LOCK"
			|| -f "$rep/vcour/fillog/LOCK"
			|| -f "$rep/vcours/cft/fillog/LOCK"
			|| -f "$rep/vcours/cft/vcours/fillog/LOCK");
}

sub CftLogs {
	my ($rep) = @_;
	my @result;
	my @logs = qw(log cft_log alog cft_loga loga cft_alog);
	for(@logs) {
		my $log = "$rep/fillog/$_";
		$log = "$rep/v_cour/fillog/$_" if(! -f $log);
		$log = "$rep/vcour/fillog/$_" if(! -f $log);
		@result = (@result,$log) if(-f $log);
	}
	return @result;
}

# les rpertoires se finissant par un 'b' sont pour les batchs
sub CftDir {
	my(%rep,@rep,$rep);
	my $dir = "/logiciel";
	opendir(DIR,$dir);
	while(defined($rep = readdir(DIR))) {
		next if($rep =~ /^(\.|\.\.)$/);
		if($rep =~ /^cft/) {
			my $file = "$dir/$rep";
			$rep{$file} = $file;
		}
	}
	closedir(DIR);
	if(open(AUTO,"</etc/auto_logiciel")) {
		while(<AUTO>) {
			$rep{"/logiciel/$1"} = "/logiciel/$1" if(/^(cft\S+)/);
		}
		close(AUTO);
	}
	for(sort keys %rep) {
		my $rep = $_;
		if($rep =~ /^(\S+)b$/) {
			my $test = $1;
			next if($rep{$test});
		}
		push(@rep,$rep);
	}
	return @rep;
}

sub InstanceOracle {
	my($host) = @_;
	my $nb = 0;
	my %instance;
	my $propre = 0;
	my @rep = ("/etc",
			   "/var/opt/oracle");
	for(@rep) {
		my $dir = $_;
		if(-d "$dir" && -f "$dir/oratab") {
			$propre = 1;
		}
	}
	if(! $propre) {
		Trace "Installation oratab non standard";
		@rep = ("/logiciel/oracle/local",
				"/logiciel/oracle_mgr/local",
				"/home/logiciel/oracle/local",
				"/home/logiciel/oracle_mgr/local",
				"/$host/logiciel/oracle/local",
				"/$host/logiciel/oracle_mgr/local");
	}
	for(@rep) {
		my $dir = $_;
		if(-d "$dir" && -f "$dir/oratab") {
			Trace "$dir/oratab" if(Debug());
			$nb++;
			open(ORATAB,"<$dir/oratab");
			while(<ORATAB>) {
				next if(/^\#/);
				next if(/^\*/);
				my $ligne = $_;
				if($ligne =~ /^(\S+):\/\S+:[YN]/) {
					my $instance = $1;
					Trace "$instance" if(Debug());
					$instance{$instance} = $ligne;
				}
			}
			close(ORATAB);
		}
	}
	if($nb>1) {
		Trace "Plusieurs oratab detectes";
	}
	my @instance;
	for(sort keys %instance) {
		@instance = (@instance,$instance{$_});
	}
	return @instance;
}

# --------------------------------------------------------
# fonctions systeme exportees
# --------------------------------------------------------

sub chkconfig {
	return standard('CHKCONFIG',@_);
}

sub at {
	return standard('AT',@_);
}

sub bdf {
	my @param = @_;
	my @result = ("Filesystem Type kbytes use avail %use Mounted\n");
	my @temp = `$CMD{$OS}{BDF} @param` if($CMD{$OS}{BDF});
	my($filesystem,$type,$size,$used,$avail,$capacity,$mountpoint);
	for(@temp) {
		next if(/^\s*Filesystem/);
		$filesystem = $1 if(/^\s*(\S+)\s*$/);
		($size,$used,$avail,$capacity,$mountpoint,$type) = ($1,$2,$3,$4,$5,"?")
			if(/^\s*(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s*$/);
		($filesystem,$size,$used,$avail,$capacity,$mountpoint,$type) = ($1,$2,$3,$4,$5,$6,"?")
			if(/^\s*(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s*$/);
		($filesystem,$type,$size,$used,$avail,$capacity,$mountpoint) = ($1,$2,$3,$4,$5,$6,$7)
			if(/^\s*(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s*$/);
		@result = (@result,"$filesystem $type $size $used $avail $capacity $mountpoint\n");
	}
	return @result;
}

sub cat {
	my @files = @_;
	my @result = ();
	while(@files) {
		my $file = shift(@files);
		if(open(FIC,"<$file")) {
			(@result) = (@result,<FIC>);
			close(FIC);
		} else {
			(@result) = (@result,"cat: cannot open $file");
		}
	}
	return @result;
}

sub cfgtest {
	my ($user,@param) = @_;
	return su("$user","cfgtest @param");
}

sub cftutil {
	my ($user,$rep,@param) = @_;
	my @result = "cftutil non trouve";
	my $code = 1;
	my $PATH_CFT = "/logiciel/$rep/filexe";
	$PATH_CFT = "$rep/filexe" if(! -x "$PATH_CFT/CFTUTIL");
	$PATH_CFT = "$rep/v_cour/filexe" if(! -x "$PATH_CFT/CFTUTIL");
	$PATH_CFT = "$rep/vcour/filexe" if(! -x "$PATH_CFT/CFTUTIL");
	if(-x "$PATH_CFT/CFTUTIL") {
		($code,@result) = su($user,"$PATH_CFT/CFTUTIL @param");
	}
	return ($code,@result);
}

sub date {
	my @param = @_;
	my $result = POSIX::strftime(@_,localtime());
	return $result;
}

sub df {
	my @param = @_;
	my @result = "Commande non supportee";
	@result = `$CMD{$OS}{DF} @param` if($CMD{$OS}{DF});
	return @result;
}

sub domainname {
	my @param = @_;
	my @result = "Commande non supportee";
	@result = `$CMD{$OS}{DOMAINNAME} @param` if($CMD{$OS}{DOMAINNAME});
	return @result;
}

sub du {
	my @param = @_;
	my @result = "Commande non supportee";
	@result = `$CMD{$OS}{DU} @param` if($CMD{$OS}{DU});
	return @result;
}

sub dspstat {
	my($user,$remote,@param) = @_;
	my($code,@result) = su("$user","dspstat @param -h $remote");
	return ($code,@result);
}

sub exportfs {
	return standard('EXPORTFS',@_);
}

sub hastat {
	if(VerifCommande('HASTAT')) {
		return standard('HASTAT',@_);
	} else {
		return standard('HASTAT2',@_);
	}
}

sub haget {
	if(VerifCommande('HAGET')) {
		return standard('HAGET',@_);
	} else {
		return standard('HAGET2',@_);
	}
}

sub hareg {
	if(VerifCommande('HAREG')) {
		return standard('HAREG',@_);
	} else {
		return standard('HAREG2',@_);
	}
}

sub haoracle {
	if(VerifCommande('HAORACLE')) {
		return standard('HAORACLE',@_);
	} else {
		return standard('HAORACLE2',@_);
	}
}

sub ifconfig {
	return standard('IFCONFIG',@_);
}

sub iostat {
	return standard('IOSTAT',@_);
}

sub ipcs {
	my @param = @_;
	my @result = "Commande non supportee";
	@result = `$CMD{$OS}{IPCS} @param` if($CMD{$OS}{IPCS});
	return @result;
}

sub ipssend {
	return standard('IPSSEND',@_);
}

sub logger {
	return standard('LOGGER',@_);
}

sub mail {
	my($message,@param) = @_;
	my $code = 1;
	if($CMD{$OS}{SENDMAIL}) {
		if(open(MAIL,"| $CMD{$OS}{SENDMAIL} @param")) {
			print MAIL $message;
			close(MAIL);
			$code = 0;
		}
	}
	return $code;
}

sub metadb {
	my @param = @_;
	my @result = "Commande non supportee";
	my $code = 1;
	if($CMD{$OS}{METADB}) {
		@result = `$CMD{$OS}{METADB} @param`;
		$code = $?;
	}
	return ($code,@result);
}

sub metaset {
	return standard('METASET',@_);
}

sub metastat {
	return standard('METASTAT',@_);
}

sub ncs {
	if($CMD{$OS}{NCS}) {
		return standard('NCS',@_);
	} else {
		my $prog = "/var/ncs/llbd";
		my @result = `$prog` if(-x $prog);
		my $code = $?;
		return ($code,@result);
	}
}

sub netstat {
	my @param = @_;
	my @result = "Commande non supportee";
	@result = `$CMD{$OS}{NETSTAT} @param` if($CMD{$OS}{NETSTAT});
	return @result;
}

sub nslookup {
	my @param = @_;
	my @result = "Commande non supportee";
	@result = `$CMD{$OS}{NSLOOKUP} @param` if($CMD{$OS}{NSLOOKUP});
	return @result;
}

sub opcagt {
	my @param = @_;
	my @result = "Commande non supportee";
	my $code = 1;
	if($CMD{$OS}{INITD} && -f "$CMD{$OS}{INITD}/opcagt") {
		my @temp;
		for(@param) {
			my $param = $_;
			$param =~ s/^-kill/stop/;
			$param =~ s/^-//;
			push(@temp,$param);
		}
		@result = `$CMD{$OS}{INITD}/opcagt @temp`;
		$code = $?;
	} elsif($CMD{$OS}{OPCAGT}) {
		@result = `$CMD{$OS}{OPCAGT} @param`;
		$code = $?;
	}
	print "@result";
	return $code;
}

sub opcsv {
	my @param = @_;
	my @result = "Commande non supportee";
	my $code = 1;
	if($CMD{$OS}{OPCSV}) {
		@result = `$CMD{$OS}{OPCSV} @param`;
		$code = $?;
	}
	print "@result";
	return $code;
}

sub opcctla {
	return standard('OPCCTLA',@_,'2>&1');
}

sub opcctlm {
	return standard('OPCCTLM',@_,'2>&1');
}

sub ovstop {
	return standard('OVSTOP',@_,'2>&1');
}

sub ovstart {
	return standard('OVSTART',@_,'2>&1');
}

sub ovstatus {
	return standard('OVSTATUS',@_,'2>&1');
}

sub oralsn {
	return standard('ORALSN',@_);
}

sub ovoracle {
	return standard('OVORACLE',@_);
}

sub opcragt {
	my @param = @_;
	my @result = "Commande non supportee";
	my $code = 1;
	if($CMD{$OS}{OPCRAGT}) {
		@result = `$CMD{$OS}{OPCRAGT} @param`;
		$code = $?;
	}
	print "@result";
	return $code;
}

sub opcackmsg {
	return standard('OPCACKMSG',@_);
}

sub opcmsg {
	my($application,$object,$msg_text,$severity,$msg_grp,$node) = @_;
	my @result = "Commande non supportee";
	my $code = 1;
	my $param = join(' ', "application=\"$application\"",
					 "object=\"$object\"",
					 "msg_text=\"$msg_text\"",
					 "severity=\"$severity\"",
					 "msg_grp=\"$msg_grp\"");
	$param = "$param node=\"$node\"" if($node);
	if(VerifCommande("OPCMSG")) {
		@result = `$CMD{$OS}{OPCMSG} $param`;
		$code = $?;
		print "@result";
		if($code != 0) {
			my $texte = "Erreur opcmsg $param";
			Trace $texte;
			$texte = "Erreur opcmsg code = " . ($code >> 8) . "-" . ($code & 255) . " : $!";
			Trace $texte;
		}
	} else {
		Trace "$msg_text";
	}
	return $code;
}

sub pmap {
	my @param = @_;
	my @result = "Commande non supportee";
	@result = `$CMD{$OS}{PMAP} @param 2>&1` if($CMD{$OS}{PMAP});
	return @result;
}

sub pstack {
	my @param = @_;
	my @result = "Commande non supportee";
	@result = `$CMD{$OS}{PSTACK} @param 2>&1` if($CMD{$OS}{PSTACK});
	return @result;
}

sub pfiles {
	my @param = @_;
	my @result = "Commande non supportee";
	@result = `$CMD{$OS}{PFILES} @param 2>&1` if($CMD{$OS}{PFILES});
	return @result;
}

sub ping {
	my @param = @_;
	my @result = "Commande non supportee";
	@result = `$CMD{$OS}{PING} @param 2>&1` if($CMD{$OS}{PING});
	return @result;
}

sub pnmstat {
	return standard('PNMSTAT',@_);
}

sub powermt {
	return standard('POWERMT',@_);
}

sub ps {
	my @param = @_;
	my @result;
	my @temp = "Commande non supportee";
	@param = (" ") if($OS eq 'NT');
	@temp = `$CMD{$OS}{PS} @param` if($CMD{$OS}{PS});
	my $code = $?;
	Trace "Erreur $CMD{$OS}{PS} @param : code $code : $!" if($OS eq 'NT' && $code!=0);
	if($OS eq 'NT') {
		for(@temp) {
			if(/^\d+\s+/) {
				my($pid,$cmd,$priority,$thread,$size,$unite,$time) = split;
				push(@result,"root $pid 1 ?? 0:0:0 ?? $time $cmd");
			} elsif(/^ID/) {
				push(@result,"UID PID PPID STIME TIME CMD");
			}
		}
	} else {
		@result = @temp;
	}
	return @result;
}

sub quota {
	return standard('QUOTA',@_);
}

sub rsh {
	return standard('RSH',@_);
}

sub sar {
	return standard('SAR',@_);
}

sub sqlplus {
	my($login,$requete) = @_;
	my @result = "Commande non supportee";
	my $ORACLE_HOME = $ENV{ORACLE_HOME} || "";
	if(-x "$ORACLE_HOME/bin/sqlplus") {
		my $out = "/var/tmp/sqlplus.$$";
		if(open(SQL,"|$ORACLE_HOME/bin/sqlplus -s $login > $out")) {
			print SQL $requete;
			close(SQL);
		} else {
			@result = (@result,"$ORACLE_HOME/bin/sqlplus : $!\n");
		}
		if(open(OUT,"<$out")) {
			@result = <OUT>;
			close(OUT);
			unlink $out if(-f $out);
		} else {
			@result = (@result,"open $out : $!\n");
		}
	}
	return @result;
}

sub sendmail {
	return standard('SENDMAIL',@_);
}

sub servOCSEPS {
	return standard('SERVOCSEPS',@_);
}

sub servlicenceOCSEPS {
	return standard('SERVLICENCEOCSEPS',@_);
}

sub su {
	my($user,@param) = @_;
	my @temp = "Commande non supportee";
	my @result;
	my $code = 1;
	my $ok = 0;
	if($CMD{$OS}{SU}) {
		@temp = timeout("$CMD{$OS}{SU} - $user -c \'@param\' 2>&1",20);
		if(grep(/timeout/,@temp)) {
			$code = 2;
		} else {
			$code = 0;
		}
	}
	for(@temp) {
		next if($ok==0 && /^\s*\*.*\*\s*$/);	# messages d'accueil
		next if(/Sun Microsystems/);			# messages d'accueil
		next if(/You have.*mail/);				# messages d'accueil
		next if(/Over disk quota/);				# messages d'accueil
		$ok = 1;
		@result = (@result,"$_");
	}
	return ($code,@result);
}

sub tail {
	my @param = @_;
	my @result = "Commande non supportee";
	@result = `$CMD{$OS}{TAIL} @param` if($CMD{$OS}{TAIL});
	return @result;
}

sub xlvmgr {
	my @param = @_;
	my @result = "Commande non supportee";
	@result = `$CMD{$OS}{XLVMGR} @param` if($CMD{$OS}{XLVMGR});
	return @result;
}

# commande Tivoli
sub wping {
	my @param = @_;
	my @result = "Commande non supportee";
	@result = timeout(". $CMD{$OS}{TIVOLI} ; wping @param",180) if($CMD{$OS}{TIVOLI});
	return @result;
}

# commande Tivoli
sub objcall {
	my @param = @_;
	my @result = "Commande non supportee";
	@result = timeout(". $CMD{$OS}{TIVOLI} ; objcall @param",180) if($CMD{$OS}{TIVOLI});
	return @result;
}

# commande Tivoli
sub wlookup {
	my @param = @_;
	my @result = "Commande non supportee";
	@result = timeout(". $CMD{$OS}{TIVOLI} ; wlookup @param",180) if($CMD{$OS}{TIVOLI});
	return @result;
}

sub scconf {
	my (@param) = @_;
	my @result = "Commande non supportee\n";
	my $code = 1;
	my $pid = open(KID, "-|");
	if ($pid) {   # parent
		@result = <KID>;
		$code=0 if close(KID);
	} else {      # child
		exec("$CMD{$OS}{SCCONF} @param")
			|| die "can't exec $CMD{$OS}{SCCONF} : $!\n";
		# NOTREACHED
	}
	my $temp = "/var/tmp/sed.tmp$pid";
	unlink($temp) if(-f $temp); # Sun laisse trainer des trucs
	return ($code,@result);
}

sub top {
	my @param = @_;
	my $debut = 1;
	my @result = ("USER PID TIME %CPU COMMAND\n");
	my(%pid,%user,%time,%pcpu,%args);
	my($tty,$pid,$user,$priority,$nice,$rss,$vsz,$state,$time,$wcpu,$pcpu,@args);
	my @temp = `$CMD{$OS}{TOP} @param` if($CMD{$OS}{TOP});
#	@temp = split(/\r/,join(' ',@temp)) if($temp[0] =~ /\r/);
	for(@temp) {
		my $ligne = $_;
#		$ligne =~ tr/a-zA-Z0-9\.\:\?_/ /cs;
#		$ligne =~ s/5C(\S+)/$1/g;
		$debut = 0 if($ligne =~ /(TTY|TT)\s+PID/);
		next if($debut || $ligne =~ /(TTY|TT)\s+PID/);
		next if($ligne =~ /defunct/);
		next if($ligne =~ /\s+$$\s+/);		# on enlve notre process
		if($ligne =~ /^\s*B\s+\d+\s+(\S+\s+\d+.*)$/) {
			($tty,$pid,$user,$priority,$nice,$rss,$vsz,$state,$time,
			 $wcpu,$pcpu,@args) = split(/\s+/,$1);
			Trace "$1\nformat 1 $pid,$user,@args" if(Debug());
		} elsif($ligne =~ /^\s*\d+\s+(\S+\s+\d+.*)$/) {
			($tty,$pid,$user,$priority,$nice,$rss,$vsz,$state,$time,
			 $wcpu,$pcpu,@args) = split(/\s+/,$1);
			Trace "$1\nformat 2 $pid,$user,@args" if(Debug());
		} elsif($ligne =~ /^\s*\S+\s+\d+/) {
			($tty,$pid,$user,$priority,$nice,$rss,$vsz,$state,$time,
			 $wcpu,$pcpu,@args) = split(/\s+/,$ligne);
			Trace "$ligne\nformat 3 $pid,$user,@args" if(Debug());
		}
		next if(! $pcpu);
		$user{$pid} = $user;
		$time{$pid} = $time;
		$pcpu{$pid} = $pcpu;
		$args{$pid} = join(' ',@args);
	}
	for(sort { $pcpu{$b} <=> $pcpu{$a} } keys %pcpu) {	# tri sur pcpu descending
		@result = (@result,"$user{$_} $_ $time{$_} $pcpu{$_} $args{$_}\n");
	}
	return @result;
}

sub vxprint {
	return standard('VXPRINT',@_);
}

sub vxdisk {
	return standard('VXDISK',@_);
}

sub vxdctl {
	return standard('VXDCTL',@_);
}

sub vxiod {
	return standard('VXIOD',@_);
}

sub vxdg {
	return standard('VXDG',@_);
}

sub ypcat {
	my @param  = @_;
	my @result = "Commande non supportee";
	@result = `$CMD{$OS}{YPCAT} @param` if($CMD{$OS}{YPCAT});
	return @result;
}

sub ypmatch {
	my @param  = @_;
	my @result = "Commande non supportee";
	@result = `$CMD{$OS}{YPMATCH} @param` if($CMD{$OS}{YPMATCH});
	return @result;
}

sub ypwhich {
	my @param  = @_;
	my @result = "Commande non supportee";
	@result = `$CMD{$OS}{YPWHICH} @param` if($CMD{$OS}{YPWHICH});
	return @result;
}

__END__
