#!/serveur/dp/bin/perl -w

# 	$Id: Filtre.pl,v 2.0 2000/07/28 09:46:37 verdiemi Exp $	

# -------------------------------------------------------------------
# Filtrage des alertes sur les disques.
#
# On stocke, sur le host concern, une mini-base contenant un
# identifiant de l'anomalie, par exemple le nom d'un filesystem.
# On y stocke la date de remonte d'alerte. Les anomalies dj
# rfrences ne donnent pas lieu  une alerte.Les enregistrements
# de plus de x jours sont ignors. Par exemple 3 jours pour le
# remplissage des disques. Donc on aura une nouvelle alerte pour une
# mme anomalie seulement tous les x jours.
#
# Ce script pourra tre utilis pour une remonte automatique dans ARS.
#
# Dure sparant 2 messages pour le remplissage des disques :
# - /var   : 24h
# - /      : 4h
# - dfaut : 72h (3 jours)
#
# Criticit des messages pour le remplissage des disques :
#	soft				minor
#	(data|implan)		major
#	(disk|work)			major
#	(vault|vaults)		major
#	dfaut				critical
# L'objet est modifi pour les disques implantation afin de leur
# envoyer directement un mail.
#
# usage : Filtre.pl <type de filtre> <paramtres>
#
#		type de filtre : 1=filesystem 2=filtre horaire
#		paramtres : selon le type de filtre  utiliser
#				1 : filesystem ou mountpoint ayant un problme
#				2 : heures de surveillance (pas de message si hors horaires)
#
# Code retour :
#		0 : ok , sinon : erreur
#
# exemples :
#		Filtre.pl 1 /export/home/disk1
#		Filtre.pl 1 xlv_casl1s2
#		Filtre.pl 2 07:00-05:00
#
# -------------------------------------------------------------------

use strict;
use AnyDBM_File;
use Fcntl ':flock';
use Fcntl;
use Time::Local;
use File::Basename;
use lib dirname($0);
use dits_def;

my $debug = Debug();

# -------------------------------------------------------------------
# Rcupration des paramtres :
# - type de l'anomalie  vrifier
# - paramtres pour le type de filtre
# -------------------------------------------------------------------
my $type_filtre = shift(@ARGV);
my $param1 = shift(@ARGV);
my $param2 = shift(@ARGV);

# -------------------------------------------------------------------
# calcul de l'heure courante
# -------------------------------------------------------------------
my($jour_cour,$mois_cour,$annee_cour,$heure_cour,$min_cour,$sec_cour) =
	split(/:/,date("%d:%m:%Y:%H:%M:%S"));
my $time_cour = timelocal($sec_cour,$min_cour,$heure_cour,
						  $jour_cour,$mois_cour-1,$annee_cour);

# -------------------------------------------------------------------
# Envoie une alerte  ITO pour un filesystem rempli
# avec dtermination de la svrit
# Pour l'implantation, on met un code objet diffrent pour envoi de mail.
# -------------------------------------------------------------------
sub message_dsk {
	my($filesystem) = @_;
	my $msg_txt = "Probleme de remplissage pour $filesystem";
	my $severity = "critical";
	my $objet = "disk";
	if($filesystem =~ /\/cale/) {
		$severity = "critical";
	} elsif($filesystem =~ /soft\d+$/) {
		$severity = "minor";
	} elsif($filesystem =~ /(data|data-|implan)\d+$/) {
		$severity = "major";
		if(opendir(DIR,$filesystem)) {
			my @dir = readdir(DIR);
			close(DIR);
			$objet = "disque implantation" if(grep(/IMPLANTATION/,@dir));
		}
	} elsif($filesystem =~ /(disk|work)\d+$/) {
		$severity = "major";
	} elsif($filesystem =~ /\/(vault|vaults)\//) {
		$severity = "major";
	}
	if($debug) {
		print "opcmsg Disk Space,$objet,$msg_txt,$severity,OS\n";
		return 0;
	}
	my $code = opcmsg("Disk Space",$objet,$msg_txt,$severity,"OS");
	return $code;
}

# -------------------------------------------------------------------
# On peut recevoir un device en paramtre. Il faut le convertir
# en mountpoint. Par contre si on recoit mountpoint et device,
# on stocke le device (cf disques de cale).
# Parcours de la base DB
# On utilise un fichier lock pour viter les problmes...
# -------------------------------------------------------------------
sub filtre_dsk {
	my($filesystem,$device) = @_;
	my $dir = OpcLog();
	my $base = "$dir/filtre_dsk.db";
	my $lock = "$dir/filtre_dsk.lock";
	my %date;
	my $code = 0;

	if($filesystem !~ /^\// || $filesystem =~ /^\/dev\//) {
		my @result = bdf();
		for(@result) {
			next if(! /$filesystem/);
			my($fs,$type,$size,$used,$avail,$capacity,$mountpoint) = split(/\s+/);
			$filesystem = $mountpoint;
			last;
		}
	}

	my $nb_heure = 72;
	$nb_heure = 24 if($filesystem eq '/var');
	$nb_heure = 4 if($filesystem eq '/');

	my $mountpoint = $device || $filesystem;

	open(LOCK,">>$lock") && flock(LOCK,LOCK_EX)
		&& tie(%date,'NDBM_File',$base,O_RDWR|O_CREAT, 0640);
	if(!$date{$mountpoint} || $time_cour<$date{$mountpoint}
	   || (($time_cour-$date{$mountpoint})/(60*60))>$nb_heure) {
		$code = message_dsk($filesystem);
		$date{$mountpoint} = $time_cour if($code == 0);
	}
	untie(%date);
	flock(LOCK,LOCK_UN);
	close(LOCK);
	return $code;
}

# -------------------------------------------------------------------
# vrifie l'heure
# on renvoie un code 1 si on est dans les horaires de surveillance
# ainsi il n'y aura pas d'acknowledge
# On regarde si heure dbut < heure fin (cas classique) ou si
# heure dbut > heure fin (par exemple de 7h a 5h le lendemain)
# -------------------------------------------------------------------
sub filtre_horaire {
	my($heures) = @_;
	my $code = 0;
	my($deb,$fin) = split(/-/,$heures);
	my ($hh_deb,$mm_deb) = split(/:/,$deb);
	my ($hh_fin,$mm_fin) = split(/:/,$fin);
	$mm_deb = 0 if(!defined($mm_deb));
	$mm_fin = 0 if(!defined($mm_fin));
	my $time_deb = timelocal(0,$mm_deb,$hh_deb,$jour_cour,$mois_cour-1,$annee_cour);
	my $time_fin = timelocal(0,$mm_fin,$hh_fin,$jour_cour,$mois_cour-1,$annee_cour);
	Trace "cour=$time_cour deb=$time_deb fin=$time_fin"if($debug);
	if(($time_deb>$time_fin
		&& ($time_cour>=$time_deb || $time_cour<=$time_fin))
	   || ($time_deb<=$time_fin
		   && ($time_cour>=$time_deb && $time_cour<=$time_fin))) {
		$code = 1;
	}
	return $code;
}

# -------------------------------------------------------------------
# Appel fonctions selon le type de filtre
# -------------------------------------------------------------------
my $code = 0;
if(! $type_filtre) {
	Trace "Parametre type de filtre requis";
	$code = 1;
} elsif($type_filtre !~ /^\d+$/) {
	Trace "Parametre type de filtre ($type_filtre) non numerique";
	$code = 1;
} elsif($type_filtre == 1) {						# filesystem
	if(! $param1) {
		Trace "Parametre <filesystem> requis";
		$code = 1;
	} else {
		if(! $param2) {
			$code = filtre_dsk($param1);
		} else {
			$code = filtre_dsk($param1,$param2);
		}
	}
} elsif($type_filtre == 2) {						# horaires
	if(! $param1) {
		Trace "Parametre <horaires> requis";
		$code = 1;
	} else {
		$code = filtre_horaire($param1);
	}
} else {
	Trace "Type de filtre ($type_filtre) non encore traite";
	$code = 1;
}

Trace "code = $code" if($debug);
exit $code;

__END__
