#!/usr/bin/sh
# OPC_WHAT_STRING="@(#)HP OpenView IT/Operations A.04.12 (01/08/99)"
###############################################################################
#
# File:         mondbfile.sh
# Description:  Monitor remaining space for autoextend DB tablespaces
# Language:     Bourne Shell
# Package:      HP OpenView IT/Operations
#
# (c) Copyright Hewlett-Packard Co. 1993 - 1999
#
###############################################################################
# set -x

PROGNAME=`basename $0`

#########################################################################
# Shell libraries
#   Set also mgmt server variables. This are needed to get the Oracle
#   variables on the mgmt server
#########################################################################

BIN_DIR=/usr/bin
OS_DIR=/usr
OPT_DIR=/opt
VAR_DIR=/var/opt
ETC_DIR=/etc/opt

# localization
LANG_C=C
LANG_SJIS=ja_JP.SJIS
LANG_EUC=ja_JP.eucJP

# for agents we may need a different root directory e.g. in a cluster
# environment else it is the root directory
#
ROOT_DIR=${ROOT_DIR:-""}

# defined in every script - the version of this script
INST_OPCVERS=A.04.12

# prerequisites:
#---------------
#
# the paths
#
#   OPT_DIR	e.g. /opt
#   VAR_DIR	e.g. /var/opt
#   ETC_DIR	e.g. /etc/opt
#
# must be defined before using this module
#
# Naming Scheme
# <prog>_<name>_DIR	1 directory e.g. OV_<name>_DIR, OPC_<name>_DIR
# <name>_PATH		several directories delimited by ":"
#
TMP_DIR=${ROOT_DIR}/tmp/opc_tmp

#-------------------------------------
# generic paths
#-------------------------------------
# /opt/OV
OV_OPT_DIR=${OV_OPT_DIR:-${ROOT_DIR}${OPT_DIR}/OV}
OPC_OPT_DIR=${OPC_OPT_DIR:-${OV_OPT_DIR}}
# /var/opt/OV
OV_VAR_DIR=${OV_VAR_DIR:-${ROOT_DIR}${VAR_DIR}/OV}
OV_VAR_SH_DIR=${OV_VAR_SH_DIR:-${OV_VAR_DIR}/share}
OPC_VAR_DIR=${OPC_VAR_DIR:-${OV_VAR_DIR}}
# /etc/opt/OV
OV_ETC_DIR=${OV_ETC_DIR:-${ROOT_DIR}${ETC_DIR}/OV}
OV_ETC_SH_DIR=${OV_ETC_SH_DIR:-${OV_ETC_DIR}/share}
OPC_ETC_DIR=${OPC_ETC_DIR:-${OV_ETC_DIR}/share}

OV_NEWCFG_DIR=${OV_NEWCFG_DIR:-${OV_OPT_DIR}/newconfig}
OV_OLDCFG_DIR=${OV_OLDCFG_DIR:-${OV_OPT_DIR}/old}

# WWW
OV_WWW_DIR=$OV_OPT_DIR/www

# /opt/OV
OV_OPT10_DIR=${OV_OPT10_DIR:-${ROOT_DIR}${OPT10_DIR}/OV}
# /var/opt/OV
OV_VAR10_DIR=${OV_VAR10_DIR:-${ROOT_DIR}${VAR10_DIR}/OV}
OV_VAR10_SH_DIR=${OV_VAR10_SH_DIR:-${OV_VAR10_DIR}/share}
# /etc/opt/OV
OV_ETC10_DIR=${OV_ETC10_DIR:-${ROOT_DIR}${ETC10_DIR}/OV}
OV_ETC10_SH_DIR=${OV_ETC10_SH_DIR:-${OV_ETC10_DIR}/share}

# where to find the minimal SD binaries
OPC_SD_DIR=${OPC_SD_DIR:-/usr/sbin}

#-------------------------------------
# OpenView paths
#-------------------------------------
OV_BIN_DIR=${OV_BIN_DIR:-${OV_OPT_DIR}/bin}
OV_TMP_DIR=${OV_TMP_DIR:-${OV_VAR_DIR}/tmp}
OV_LOG_DIR=${OV_LOG_DIR:-${OV_VAR_DIR}/log}
OV_CFG_DIR=${OV_CFG_DIR:-${OV_ETC_DIR}/share/conf}
OV_ALT_CFG_DIR=${OV_ALT_CFG_DIR:-${OV_ETC_DIR}/share/\#conf}

#-------------------------------------
# ITO Server Paths
#-------------------------------------
# /usr/OV/bin/OpC
OPCSVR_BIN_DIR=${OPCSVR_BIN_DIR:-${OPC_OPT_DIR}/bin/OpC}
# /usr/OV/bin/OpC/install
OPCSVR_INST_DIR=${OPCSVR_INST_DIR:-${OPCSVR_BIN_DIR}/install}
# /opt/OV/lib
OPCSVR_LIB_DIR=${OPCSVR_LIB_DIR:-${OPC_OPT_DIR}/lib}
# /opt/OV/nls
OPCSVR_NLS_DIR=${OPCSVR_NLS_DIR:-${OPCSVR_LIB_DIR}/nls}
# /opt/OV/include
OPCSVR_INC_DIR=${OPCSVR_INC_DIR:-${OPC_OPT_DIR}/include}
# /usr/OV/bin/OpC/utils
OPCSVR_UTIL_DIR=${OPCSVR_UTIL_DIR:-${OPCSVR_BIN_DIR}/utils}
# /usr/OV/bin/OpC/agtinstall
OPCSVR_AGTINST_DIR=${OPCSVR_AGTINST_DIR:-${OPCSVR_BIN_DIR}/agtinstall}
# /etc/opt/OV/share/lrf
OPCSVR_OPC_DIR=${OPCSVR_OPC_DIR:-${OPC_OPT_DIR}/OpC}
OPCSVR_NEWCFG_DIR=${OPCSVR_NEWCFG_DIR:-${OV_NEWCFG_DIR}/OpC}
OPCSVR_OLDCFG_DIR=${OPCSVR_OLDCFG_DIR:-${OV_OLDCFG_DIR}/OpC}
OPCSVR_HELP_DIR=${OPCSVR_HELP_DIR:-${OV_VAR_DIR}/share/help}

# /etc/opt/OV directories
OPCSVR_LRF_DIR=${OPCSVR_LRF_DIR:-${OPC_ETC_DIR}/lrf}
OPCSVR_REG_DIR=${OPCSVR_REG_DIR:-${OPC_ETC_DIR}/registration}

# /var/opt/OV/share/databases
OPCSVR_DB_DIR=${OPCSVR_DB_DIR:-${OPC_VAR_DIR}/share/databases/OpC}
# /var/opt/OV/share/databases/OpC/mgd_node/vendor
OPCSVR_VEND_DIR=${OPCSVR_DB_DIR}/mgd_node/vendor
# /var/opt/OV/share/databases/OpC/mgd_node/customer
OPCSVR_CUST_DIR=${OPCSVR_DB_DIR}/mgd_node/customer

# dynamic paths
OPCSVR_TMP_DIR=${OPCSVR_TMP_DIR:-${OPC_VAR_DIR}/share/tmp/OpC}
OPCSVR_LOG_DIR=${OPCSVR_LOG_DIR:-${OPC_VAR_DIR}/log/OpC}
OPCSVR_CFG_DIR=${OPCSVR_CFG_DIR:-${OPC_ETC_DIR}/conf/OpC}
OPCSVR_APPL_DIR=${OPCSVR_APPL_DIR:-${OPC_VAR_DIR}/share/tmp/OpC_appl}
OPCSVR_MAN_DIR=${OPCSVR_MAN_DIR:-${OPC_OPT_DIR}/man}

OPCSVR_INFO_F=${OPCSVR_INFO_F:-${OPCSVR_INST_DIR}/opcsvinfo}

# Config file needed by distributed GUI client
OPC_GUICLT_MSV_F=${OPC_GUICLT_MSV_F:-${OV_CFG_DIR}/opc_guiclt_msv}
OPC_GUICLT_TMP_MSV_F=${OPC_GUICLT_TMP_MSV_F:-/tmp/opc_guiclt_msv}

# DB config file
OVDB_CONFIG_FILE=${OVDB_CONFIG_FILE:-${OV_CFG_DIR}/ovdbconf}

#-------------------------------------
# ITO Agent Paths
#-------------------------------------
# /usr/OV/bin/OpC/s700
OPCAGT_BIN_DIR=${OPCAGT_BIN_DIR:-${OPC_OPT_DIR}/bin/OpC}
OPCAGT_LIB_DIR=${OPCAGT_LIB_DIR:-${OPC_OPT_DIR}/lib}
OPCAGT_NLS_DIR=${OPCAGT_NLS_DIR:-${OPCAGT_LIB_DIR}/nls}
OPCAGT_INC_DIR=${OPCAGT_INC_DIR:-${OPC_OPT_DIR}/include}
# /usr/OV/bin/OpC/s700/utils
OPCAGT_UTIL_DIR=${OPCAGT_UTIL_DIR:-${OPCAGT_BIN_DIR}/utils}
# /usr/OV/bin/OpC/s700/install
OPCAGT_INST_DIR=${OPCAGT_INST_DIR:-${OPCAGT_BIN_DIR}/install}

# /var/opt/OV/bin/OpC/monitor
OPCAGT_MON_DIR=${OPCAGT_MON_DIR:-${OPC_VAR_DIR}/bin/OpC/monitor}
# /opt/OV/bin/OpC/actions
OPCAGT_ACT_DIR=${OPCAGT_ACT_DIR:-${OPC_VAR_DIR}/bin/OpC/actions}
# /var/opt/OV/bin/OpC/cmds
OPCAGT_CMD_DIR=${OPCAGT_CMD_DIR:-${OPC_VAR_DIR}/bin/OpC/cmds}

# dynamic paths
OPCAGT_LOG_DIR=${OPCAGT_LOG_DIR:-${OPC_VAR_DIR}/log/OpC}
OPCAGT_CFG_DIR=${OPCAGT_CFG_DIR:-${OPC_VAR_DIR}/conf/OpC}
OPCAGT_TMP_DIR=${OPCAGT_TMP_DIR:-${OPC_VAR_DIR}/tmp/OpC}
OPCAGT_TMPBIN_DIR=${OPCAGT_TMPBIN_DIR:-${OPCAGT_TMP_DIR}/bin}
OPCAGT_TMPCFG_DIR=${OPCAGT_TMPCFG_DIR:-${OPCAGT_TMP_DIR}/conf}
OPCAGT_NEWCFG_DIR=${OPCAGT_NEWCFG_DIR:-${OV_NEWCFG_DIR}/OpC}
OPCAGT_OLDCFG_DIR=${OPCAGT_OLDCFG_DIR:-${OV_OLDCFG_DIR}/OpC}

# info files
OPCAGT_NDINFO_F=${OPCAGT_NDINFO_F:-${OPCAGT_CFG_DIR}/nodeinfo}
OPCAGT_INFO_F=${OPCAGT_INFO_F:-${OPCAGT_INST_DIR}/opcinfo}
OPCAGT_NETLS_F=${OPCAGT_NETLS_F:-${OPCAGT_CFG_DIR}/opcnetls}
OPCAGT_MGRCONF_F=${OPCAGT_MGRCONF_F:-${OPCAGT_CFG_DIR}/mgrconf}
OPCAGT_UPD_F=${OPCAGT_UPD_F:-${OPCAGT_TMP_DIR}/update}
OPCAGT_CLIENT_F=${OPCAGT_CLIENT_F:-${OPCAGT_INST_DIR}/cfg.clients}
OPCAGT_MGMTSV_F=${OPCAGT_MGMTSV_F:-${TMP_DIR}/mgmt_sv.dat}
#
# commands
#
OPCMON=${OPCAGT_BIN_DIR}/opcmon
OPCMSG=${OPCAGT_BIN_DIR}/opcmsg

VUE_DIR=/etc/vue
VUE_USR_DIR=/usr/vue
VUE_ETC_DIR=/etc/vue
VUE_VAR_DIR=/var/vue
CDE_USR_DIR=/usr/dt
CDE_ETC_DIR=/etc/dt
CDE_VAR_DIR=/var/dt

#-------------------------------------
# paths
#-------------------------------------
SYSTEM_PATH=${SYSTEM_PATH:-"/bin:/usr/bin:/sbin:/usr/sbin:/etc:/usr/etc:/usr/ucb"}
OV_PATH=${OV_BIN_DIR}
OPCSVR_PATH=${OPCSVR_BIN_DIR}:${OPCSVR_AGTINST_DIR}:${OPCSVR_INST_DIR}
OPCAGT_PATH=${OPCAGT_BIN_DIR}:${OPCAGT_INST_DIR}:${OPCAGT_MON_DIR}:${OPCAGT_ACT_DIR}:${OPCAGT_CMD_DIR}

# program name
APPLNAME=ITO
LONG_APPLNAME="HP OpenView IT/Operations 4.0"

# common umask value
umask 022

# print_msg() updates EXIT_CODE automatically using
# this utility function for script exit value.
# it keeps the worst exit code in ${EXIT_CODE}
#--------------------------------------------------
SUCCESS=0
FAILURE=1
WARNING=2

EXIT_CODE=${SUCCESS}

eval_exit()
{
  exit_new=$1

  if [ ${exit_new} -eq ${FAILURE} ]
  then
    EXIT_CODE=${FAILURE}
  elif [ ${exit_new} -eq ${WARNING} -a ${EXIT_CODE} -eq ${SUCCESS} ]
  then
    EXIT_CODE=${WARNING}
  fi

  return 0
}

#-------------------------------
#
# print_msg: print message to stderr/stdout and optional to logfile
#
#-------------------------------
# usage: print_msg <mode> <messages> ...
#
# <mode>: ( <type> | <fd> | <type><fd> | "." )
#
#        <type>:     use this type with <fd> = 2
#        <fd>:       use this fd with <type> = I
#        <type><fd>: use type and fd
#        ".":        use empty prefix and fd of last call
#
# <type>: ( "I" | "E" | "W" | "N" | "U" )
#          defines message prefix:
#          I)nfo:    "         "
#          E)rror:   "ERROR:   "
#          W)arning: "WARNING: "
#          N)ote:    "NOTE:    "
#          U)sage:   "USAGE:   "
#
# <fd>: file descriptor for output (2 = stderr, 1 = stdout)
#
# <messages>: the first message argument will be printed with
#             the selected prefix. Optionally following message
#             arguments will be printed in seperate lines with
#             an empty prefix.
# 
#-------------------------------
# examples:
#
# 1.) print_msg W "Something went wrong"
#
#     output to stderr: WARNING: Something went wrong
#
# 2.) print_msg N1 "Be aware that"
#     print_msg .  "something has happened"
#
#     output to stdout: NOTE:    Be aware that
#                     :          something has happened
#
# 3.) print_msg E2 "this is" "an" "error"
#
#     output to stderr: ERROR:   this is
#                     :          an
#                     :          error
#
#-------------------------------
# logging:
#
# default: logging is disabled
# enable logging with:  enable_logging <logfile>
# disable logging with: disable_logging
#
# if logging is enabled, all output of print_msg
# is also printed into the specified logfile.
#    
#--------------------------------
# exit codes:
#
# print_msg calls the shlib_misc:eval_exit() function
# with an appropriate parameter 
# in order to keep the EXIT_CODE up to date
#
#--------------------------------
MSG_LOGFILE=""

MSG_INFO="I"
MSG_ERROR="E"
MSG_WARN="W"
MSG_NOTE="N"
MSG_USAGE="U"
MSG_QUERY="Q"
MSG_LOG="L"    # goes to log only if enabled
MSG_SAME="."

#
# prefixes must have length 9
#                 123456789
      MSG_PREFIX="         "
 MSG_PREFIX_INFO="         "
MSG_PREFIX_ERROR="ERROR:   "
 MSG_PREFIX_WARN="WARNING: "
 MSG_PREFIX_NOTE="NOTE:    "
MSG_PREFIX_USAGE="USAGE:   "
MSG_PREFIX_QUERY="-------> "
MSG_PREFIX_LOG="LOG:     "

MSG_FD=2

print_msg()
{
  mode="$1"; shift

  if [ "$mode" = "$MSG_SAME" ]
  then
    type="$MSG_SAME"
  else
    type=`echo "$mode" | sed 's/^[^A-Z]*\([A-Z]\).*$/\1/'`
    MSG_FD=`echo $mode"_2" | sed 's/^[^0-9]*\([0-9][0-9]*\).*$/\1/'`
  fi

  prefix="$MSG_PREFIX"

  # set prefix
  case "$type" in
    $MSG_INFO)
      prefix="$MSG_PREFIX_INFO"
      ;;
    $MSG_ERROR)
      prefix="$MSG_PREFIX_ERROR"
      ;;
    $MSG_WARN)
      prefix="$MSG_PREFIX_WARN"
      ;;
    $MSG_NOTE)
      prefix="$MSG_PREFIX_NOTE"
      ;;
    $MSG_USAGE)
      prefix="$MSG_PREFIX_USAGE"
      ;;
    $MSG_QUERY)
      prefix="$MSG_PREFIX_QUERY"
      ;;
    $MSG_LOG)
      prefix="$MSG_PREFIX_LOG"
      ;;
    $MSG_SAME)
      prefix="$MSG_PREFIX"
      ;;
  esac

  for msg in "$@" ; do
    # concatenate with prefix
    msg="$prefix$msg"
    prefix="         "
  
    if [ "$type" != "$MSG_LOG" ]
    then
      # output message
      echo "$msg" >&$MSG_FD
    fi
  
    # if logging is enabled log the message
    if [ -n "$MSG_LOGFILE" ]
    then
      echo "$msg" >> ${MSG_LOGFILE}
    fi

  done

  # set EXIT_CODE
  case "$type" in
    $MSG_ERROR)
      eval_exit ${FAILURE}
      ;;
    $MSG_WARN)
      eval_exit ${WARNING}
      ;;
  esac
}

enable_logging()
{
  MSG_LOGFILE="$1"
}

disable_logging()
{
  MSG_LOGFILE=""
}

# variable defines
#-----------------
PASSWD_F=/etc/passwd
PASSWD_F_BAK=/etc/passwd.opc_bak
GROUP_F=/etc/group
GROUP_F_BAK=/etc/group.opc_bak

# functions in this module
#-------------------------
#

PWD_PASSWD=2
PWD_UID=3
PWD_GID=4
PWD_DESCR=5
PWD_HOME=6
PWD_SHELL=7
# return values
#   0 - ok 
#   1 - /etc/passwd not found
#   2 - user not found
#-----------------------------------
get_passwd_info()
{
  user=$1
  key=$2

  if [ -f $PASSWD_F ]
  then
    if [ `grep -c "^+\{0,1\}${user}:" $PASSWD_F` -eq 1 ]
    then
      INFO=`grep "^+\{0,1\}${user}:" $PASSWD_F | awk 'BEGIN{FS=":"}{print $'${key}'}'`
      echo "${INFO}"
    else 
      return 2
    fi
  else
    return 1
  fi
  return 0
}

get_info()
{
  _file=$1
  _field=$2

  if [ ! -f $_file ]
  then
    return 1
  fi
  awk '$1 == field {for ( i=2; i<=NF; i++ ){printf("%s",$i);if (i < NF) printf(" ")}}' field=$_field $_file
  return $?
}

###############################################################################
# Begin of shlib_oracle
###############################################################################

# Dependencies

# File used to check for valid location and other DB variables
#-------------------------------------------------------------
ORA_DBMS_FILE=bin/sqlplus
ORA_USER=oracle
DBUSER=oracle
HOMEVAR=ORACLE_HOME
DATABASE=Oracle
OPC_DB_STOP_CMD="su ${ORA_USER} -c"
OPCDBPWD=${OPCSVR_BIN_DIR}/opcdbpwd

#----------------------------------------------------
#     Functions
#----------------------------------------------------

# Help function for read_ovdbora to read value of a parameter ignoring #...
# Parameters: $1 - Name of parameter
read_ovdbconf_parm()
{
  grep -v "^[ 	]*#" "${OVDB_CONFIG_FILE}" |grep "^[ 	]*$1[ 	]" | \
     awk '{print $2}'
  return 0
}
  
# Read the ovdbora configuration file
# Returns: 0 - OK
#          1 - couldn't read file
#------------------------------------------------------------------------------
read_ovdbora()
{
  # Check if new config file exists and use it if so
  if [ ! -f "${OVDB_CONFIG_FILE}" ]
  then
    print_msg E2 "Could not open configuration file ${OVDB_CONFIG_FILE}."
    return 1
  fi

  OV_O_SID=`read_ovdbconf_parm ORACLE_SID`
  if [ "${OV_O_SID}" != "" ]
  then
    ORACLE_SID="${OV_O_SID}"
    export ORACLE_SID
  fi

  OV_O_HOME=`read_ovdbconf_parm ORACLE_HOME`
  if  [ "${OV_O_HOME}" != "" ]
  then
    ORACLE_HOME="${OV_O_HOME}"
    export ORACLE_HOME
  fi # OV_O_HOME

  OV_O_BASE=`read_ovdbconf_parm ORACLE_BASE`
  if  [ "${OV_O_BASE}" != "" ]
  then
    ORACLE_BASE="${OV_O_BASE}"
    export ORACLE_BASE
  fi # OV_O_BASE

  OV_DBA_USER=`read_ovdbconf_parm DBA_USER`
  if [ -z "${OV_DBA_USER}" ]
  then
    OV_DBA_USER="${ORA_USER}"
    export OV_DBA_USER
  else
    ORA_USER="${OV_DBA_USER}"
    DBUSER="${OV_DBA_USER}"
    export ORA_USER DBUSER OV_DBA_USER
  fi

  OV_NLS_LANG=`read_ovdbconf_parm NLS_LANG`
  if  [ "${OV_NLS_LANG}" != "" ]
  then
    NLS_LANG="${OV_NLS_LANG}"
    export NLS_LANG
  fi # OV_NLS_LANG

  OV_DATA_DIR=`read_ovdbconf_parm DATA_DIR`
  OV_ADM_DIR=`read_ovdbconf_parm ADMIN_DIR`
  OV_OS_PREFIX=`read_ovdbconf_parm OS_AUTHENT_PREFIX`
  OV_CHARSET=`read_ovdbconf_parm CHARACTER_SET`
  ITO_DATADIR=`read_ovdbconf_parm ITO_DATADIR`
  
  export OV_DATA_DIR OV_ADM_DIR OV_OS_PREFIX OV_CHARSET
  return 0
}

# -------------------------------------
# Functions for determining ORACLE_HOME
# -------------------------------------

chk_and_set_orahome()
{
  dir=$1

  if [ -x ${dir}/${ORA_DBMS_FILE} ]
  then
    ORACLE_HOME=${dir}
    export ORACLE_HOME
    return 0
  fi

  return 1
}

success_msg_ora()
{
  silent=$1

  [ -z "${silent}" ] && \
    print_msg I "Determined variable ORACLE_HOME as ${ORACLE_HOME}."

  return 0
}

# --------------------------------------------------------------
# Function to retrieve the ORACLE_HOME variable. After successful return
# ORACLE_HOME is set.
# Params: $1 - Any non-empty string to prevent success messages from
#              being printed [optional]
# Returns: 0 - OK
#          1 - couldn't get the variable
#-------------------------------------------------------------------------

get_orahome()
{
  silent=$1

  #
  # First check whether it is already set. Best case.
  #
  if [ -n "${ORACLE_HOME}" ]
  then
    chk_and_set_orahome "${ORACLE_HOME}"
    if [ $? -eq 0 ]
    then
      success_msg_ora ${silent} && return 0
    else
      print_msg W2 "Already set ORACLE_HOME value (${ORACLE_HOME}) seems to be wrong."
      print_msg .  "Will try to find correct value."
    fi
  fi
  #
  # Second, check default locations
  #
  for DIR in ${DFLT_ORAHOME}
  do
    chk_and_set_orahome "${DIR}" && success_msg_ora ${silent} && return 0
  done

  #
  # Third, try the user's home directory
  #
  ORA_HOMEDIR=`get_passwd_info ${DBUSER} ${PWD_HOME}`
  case $? in
   0) chk_and_set_orahome "${ORA_HOMEDIR}" && success_msg_ora ${silent} && return 0
      return 1 ;;
   1) print_msg W2 "/etc/passwd is not readable."
      return 1 ;;
   2) print_msg W2 "User ${DBUSER} not found in ${PASSWD_F}."
      print_msg .  "${DATABASE} Database maybe not yet installed"
      print_msg .  "or ${DBUSER} is controlled by NIS/NIS+."
      return 1 ;;
  esac

  return 1
}

# Check for an installed Oracle DB, set ORACLE_HOME
# Params: $1 - Any non-empty string to prevent success messages from
#              being printed (errors are printed !) [optional]
#         $2 - db_configured. IF $2 is "1" then ovdbconf MUST exist and
#              errors will be returned if ORACLE_HOME, ORACLE_SID or
#              NLS_LANG are not set and not in ovdbconf
# Returns: 0 - OK
#          1 - error occured
#------------------------------------------------------------------------------
chk_oracle()
{
silent="$1"
db_configured="$2"

# Avoid multiple calls
if [ "$chk_oracle_called" = "y" ]
then
  return "${chk_oracle_result}"
fi
chk_oracle_called="y"
chk_oracle_result=1

ORA_USER=oracle

if [ -f "${OVDB_CONFIG_FILE}" ]
then
  # Read values from config file
  read_ovdbora && success_msg_ora "${silent}"

  # Check if user exists
  id -u ${OV_DBA_USER} >/dev/null 2>&1
  if [ $? -ne 0 ]
  then
    print_msg E2 "Oracle user ${OV_DBA_USER} does not exist."
    return 1
  fi

  # It seems we have Oracle
  HOMEVAR=ORACLE_HOME
  DATABASE=Oracle
else
  # Check if user exists
  id -u ${ORA_USER} >/dev/null 2>&1
  if [ $? -ne 0 ]
  then
    print_msg E2 "Oracle user ${ORA_USER} does not exist."
    return 1
  fi

  # It seems we have Oracle
  DBUSER=${ORA_USER}
  HOMEVAR=ORACLE_HOME
  DATABASE=Oracle

  if [ "${db_configured}" = "1" ]
  then
    # ovdbconf MUST exist !
    print_msg E2 "Could not open configuration file ${OVDB_CONFIG_FILE}."
    print_msg .  "Please check if the database is configured properly."
    print_msg .  "Please run opcdbsetup (again). If you already configured a"
    print_msg .  "database, you can use the existing database when asked."
    return 1
  fi

  #Get Oracle values
  get_orahome "${silent}"
  if [ $? -ne 0 ]
  then
    print_msg E2 "Cannot determine ORACLE_HOME. Please set ORACLE_HOME manually."
    return 1
  fi

  if [ -z "${ORACLE_BASE}" ]
  then
    O1=`dirname $ORACLE_HOME`
    ORACLE_BASE=`dirname $O1`
    export ORACLE_BASE
  fi

  # set other defaults
  OV_DBA_USER="${ORA_USER}"
  export OV_DBA_USER
fi

# Determine DB connection
DB_CONNECT=`get_info ${OPCSVR_INFO_F} DATABASE`
export DB_CONNECT

# Set start/stop variables
DBSTART=dbstart
DBSTOP=dbshut
DB_START_CMD="${ORACLE_HOME}/bin/${DBSTART}"
DB_STOP_CMD="${ORACLE_HOME}/bin/${DBSTOP}"
DB_IMM_STOP_CMD="${ORACLE_HOME}/bin/${DBSTOP}"
DBSTART_GREP="^[^#]*${ORA_USER}.*${DBSTART}"
DBSTOP_GREP="^[^#]*${ORA_USER}.*${DBSTOP}"
OPC_DB_STOP_CMD="su ${ORA_USER} -c"

# Get Oracle Version
ORACLE_REV=7.1.6
if [ -x ${ORACLE_HOME}/bin/sqldba ]
then
  ORACLE_REV=`${ORACLE_HOME}/bin/sqldba -? | grep "Server Release" | \
    awk '{ print $4 }' | awk -F. '{ print $1"."$2"."$3 }'`
else
  if [ -x ${ORACLE_HOME}/bin/svrmgrl ]
  then
    ORACLE_REV=`${ORACLE_HOME}/bin/svrmgrl -? </dev/null | grep "Server Release" | \
      awk '{ print $4 }' | awk -F. '{ print $1"."$2"."$3 }'`
  else
     print_msg E2 "Can't find DB Administration Tool (sqldba or svrmgrl)."
     return 1
  fi
fi

error_occured=0
# Check also ORACLE_HOME if db_configured is 1
if [ "${db_configured}" = "1" ]
then
  # ORACLE_HOME
  if [ -z "${ORACLE_HOME}" ]
  then
    print_msg E2 "ORACLE_HOME variable is not set nor is it in the config file"
    print_msg .  "${OVDB_CONFIG_FILE}."
    print_msg .  "Please set it correct and re-run the program."
    error_occured=1
  fi
fi

# To avoid problems, set ORACLE_SID
if [ -z "${ORACLE_SID}" ]
then
    ORACLE_SID=openview
    print_msg W2 "ORACLE_SID variable is not set nor is it in the config file"
    print_msg .  "${OVDB_CONFIG_FILE}."
    print_msg .  "Using default: ${ORACLE_SID}."
fi

# To avoid problems, set NLS_LANG
if [ -z "${NLS_LANG}" ]
then
    case "${OV_CHARSET}" in
      JA16SJIS)
        NLS_LANG=japanese_japan.JA16SJIS
        ;;
      WE8ISO8859P1)
        NLS_LANG=american_america.WE8ISO8859P1
        ;;
      *)
        NLS_LANG=american_america.WE8ISO8859P1
        print_msg W2 "Cannot determine NLS_LANG. Using default: ${NLS_LANG}."
        ;;
    esac
fi

if [ "${error_occured}" = "1" ]
then
  return 1
fi

chk_oracle_result=0
return 0
}

# Call Oracle DBA utility (sqldba or svrmgrl)
# Params: $1 - Filename of script to be executed
#         $2 - user as which sqldba should be called [optional]
#              Default: $OV_DBA_USER
#         $3 - Any non-empty string to prevent output [optional]
# Returns: 0 - OK
#          1 - error occured
call_sqldba()
{
  silent_dba=${3:-""}
  CALL_USER=${2:-""}
  if [ "${CALL_USER}" = "" ]
  then
    CALL_USER=${OV_DBA_USER:-oracle}
  fi

  #Changed for Oracle 7.3: Determine DB Administration Tool (svrmgrl or sqldba)
  if [ ${DBA_TOOL_NAME:-""} = "" ]
  then
    if [ -x ${ORACLE_HOME}/bin/sqldba ]
    then
        DBA_TOOL="${ORACLE_HOME}/bin/sqldba lmode=y"
        DBA_TOOL_NAME=sqldba
        DBA_ERR_GREP="^DBA-"
    else
      if [ -x ${ORACLE_HOME}/bin/svrmgrl ]
      then
        DBA_TOOL=${ORACLE_HOME}/bin/svrmgrl
        DBA_TOOL_NAME=svrmgrl
        DBA_ERR_GREP="MGR-"
      else
        print_msg E2 "Can't find DB Administration Tool (sqldba or svrmgrl)."
        return 1
      fi
    fi
  fi
  
  DBA_OUTPUT=${DBA_OUTPUT:-"/tmp/dbaout.$$"}
  rm -f ${DBA_OUTPUT}
  if [ `id -n -u` = "${CALL_USER}" ]
  then
    echo "@$1\\\\nexit" | ${DBA_TOOL} >${DBA_OUTPUT} 2>&1
    RET=$?
  else
    su ${CALL_USER} -c "echo @$1\\\\nexit | ${DBA_TOOL}" >${DBA_OUTPUT} 2>&1
    RET=$?
  fi
  
  if [ ${RET} -ne 0 ]
  then
    [ -z "${silent_dba}" ] && print_msg E2 "Error occured calling ${DBA_TOOL_NAME}." \
                 "See file ${DBA_OUTPUT} for more details."
    return 1
  fi

  grep ${DBA_ERR_GREP} ${DBA_OUTPUT} >/dev/null
  if [ $? -ne 1 ]
  then
    [ -z "${silent_dba}" ] && print_msg E2 "Error occured calling ${DBA_TOOL_NAME}." \
                 "See file ${DBA_OUTPUT} for more details."
    return 1
  fi
  return 0
}

# Call Oracle DBA utility (sqldba or svrmgrl) and check for ORA- errors
# Params: $1 - Filename of script to be executed
#         $2 - user as which sqldba should be called [optional]
#              Default: $OV_DBA_USER
#         $3 - Any non-empty string to prevent output [optional]
# Returns: 0 - OK
#          1 - error occured calling sqldba
#          2 - SQL errors in output of sqldba
call_sqldba_chk()
{
  silent_dba_chk=${3:-""}
  call_sqldba ${1:-""} ${2:-""} ${3:-""}
  if [ $? -ne 0 ]
  then
    return 1
  fi

  # Check for oracle errors
  ERRORS=`grep "ORA-" ${DBA_OUTPUT} |wc -l`
  if [ "${ERRORS}" -ne "0" ]
  then
    if [ -z "${silent_dba_chk}" ]
    then
      print_msg E2 "Oracle error occured calling ${DBA_TOOL_NAME}."
      print_msg .  "See file ${DBA_OUTPUT} for more details."
    fi
    return 2
  fi

  return 0
}

###############################################################################
# end of shlib_oracle
###############################################################################

# variable defintions
#-------------------
# convention:
# all directories related to MC/SG and ITO are named
# OPCSG_*_DIR and file OPCSG_*_F
#
OPCSG_LOCAL_INFO_F=${OPCSVR_CFG_DIR}/opcsginfo
OPCSG_OPT_DIR=/opt/OV/OpC_SG
OPCSG_SHARED_INFO_F=${OPCSG_OPT_DIR}${OPCSVR_CFG_DIR}/mgmt_sv/opcsginfo
OPCSG_NODES_DIR=${OPCSVR_CFG_DIR}/mgmt_sv/opcsgnodes
OPCSG_TMPL_DIR=${OPCSVR_CFG_DIR}/mgmt_sv/opcsgtemplates

# the ITO SG package name is always ITO
#
OPCSG_PKG=OpC

# MC/SG SD products and filesets
OPCSG_PROD_CLUST=Cluster-Monitor
OPCSG_PROD_PKG=Package-Manager
OPCSG_FSET_CORE=CM-CORE
OPCSG_FSET_PKG=CM-PKG

# functions
#----------
# convention:
# functions are named opcsg_*
#

# check if ITO sg package is installed already
# return 0 or 1
#---------------------------------------
opcsg_check_local()
{
  if [ -f ${OPCSG_LOCAL_INFO_F} ]
  then
    if [ -n "`cat $OPCSVR_INFO_F | grep '^OPC_SG_NNM_STARTUP *TRUE'`" ]
    then
      return 1
    else
      return 0
    fi
  else
    return 1
  fi
}

# check if ITO sg package is installed already
# on the shared disk
# return 0 or 1
#---------------------------------------
opcsg_check_shared()
{
  if [ -f ${OPCSG_SHARED_INFO_F} ]
  then
    return 0
  else
    return 1
  fi
}

#################
# Local functions
#################
# Function free_space:
# Determine free space on specified file system
free_space()
{
  if [ "`${BDF_CMD} $1 | wc -l`" -eq "3" ]
  then
    # line break in output
    FREE_SPACE=`${BDF_CMD} $1 | awk '{ print $3 }' | sed '/used/d'`
  else
    FREE_SPACE=`${BDF_CMD} $1 | awk '{ print $4 }' | sed '/avail/d'`
  fi

  if [ "$FREE_SPACE" = "" ]
  then
    print_msg E2 "Cannot get free disk space."
    RETVAL=1
    return 1
  fi

  return 0
}

#################
# Script body
#################

# Define variables
MON_OBJ=${MON_OBJ:-mondbfile}
BDF_CMD=bdf
RETVAL=0
FILES_FOUND=0

SPOOL_FILE=/tmp/mondbfile.lst
RESULT_FILE=/tmp/mondbfile2
SQL_FILE=/tmp/mondbfile.sql
DBA_OUTPUT=/tmp/mondbfile.out

# Check, if on not active MC/SG agent -> No error
opcsg_check_local
if [ $? -eq 0 ]
then
  opcsg_check_shared
  if [ $? -ne 0 ]
  then
    # Dummy opcmon, to produce no error
    ${OPCMON} "$MON_OBJ=1"
    exit 0
  fi
fi

if [ ${TRACE:-0} -eq 1 ]
then
  date >>/tmp/mondbfile.log
fi

# Create the SQL script
echo "connect internal;" > ${SQL_FILE}
echo "set echo off" >> ${SQL_FILE}
echo "spool $SPOOL_FILE" >> ${SQL_FILE}
echo "select d.bytes, e.maxextend, e.inc, d.name" >> ${SQL_FILE}
echo "from   v\$datafile d, filext\$ e" >> ${SQL_FILE}
echo "where  d.file# = e.file#;" >> ${SQL_FILE}
echo "spool off;" >> ${SQL_FILE}
echo "exit" >> ${SQL_FILE}

# Check and set Oracle variables
chk_oracle SILENT
if [ $? -ne 0 ]
then
  exit 1
fi

# Run the file
call_sqldba_chk ${SQL_FILE}
if [ $? -ne 0 ]
then
  exit 1
fi

# Remove not needed lines from the spool file
grep '[0-9][0-9]* *[0-9][0-9]* *[0-9][0-9]*' ${SPOOL_FILE} > ${RESULT_FILE}

# Get the DB block size
BLOCK_SIZE_B=`grep -i "^[ 	]*db_block_size" ${ORACLE_HOME}/dbs/init${ORACLE_SID}.ora |cut -f 2 -d =`
if [ "${BLOCK_SIZE_B}" = "" ]
then
  BLOCK_SIZE_B=8192
fi
BLOCK_SIZE=`expr $BLOCK_SIZE_B / 1024`

# In loop process each data file with autoextend
cat ${RESULT_FILE} |while read line
do
  FILES_FOUND=1
  REST_DS=0
  REST_TS=0
  SIZE_BYTE=`echo $line |awk '{print $1}'`
  SIZE=`expr $SIZE_BYTE / 1024`
  MAX_BLOCKS=`echo $line |awk '{print $2}'`
  MAX_SIZE=`expr $MAX_BLOCKS \* $BLOCK_SIZE`
  NEXT_BLOCKS=`echo $line |awk '{print $3}'`
  NEXT_SIZE=`expr $NEXT_BLOCKS \* $BLOCK_SIZE`
  FILE=`echo $line |awk '{print $4}'`

  # Remaining space on disk - next
  PARENT_DIR=`dirname $FILE`
  free_space $PARENT_DIR
  if [ $? -eq 0 ]
  then
    REST_DS=`expr $FREE_SPACE - $NEXT_SIZE`
    ${OPCMON} "$MON_OBJ=$REST_DS" -object "DS:$FILE"
    if [ $? -ne 0 ]
    then
      RETVAL=1
    fi
  fi

  # Remaining space until max data file size - next
  REST_IN_TS=`expr $MAX_SIZE - $SIZE`
  REST_TS=`expr $REST_IN_TS - $NEXT_SIZE`
  ${OPCMON} "$MON_OBJ=$REST_TS" -object "TS:$FILE"
  if [ $? -ne 0 ]
  then
    RETVAL=1
  fi

  if [ ${TRACE:-0} -eq 1 ]
  then
    echo "$SIZE $MAX_SIZE $NEXT_SIZE $REST_DS $REST_TS $FILE" >>/tmp/mondbfile.log
  fi
done

if [ "$RETVAL" -eq 0 ]
then
  rm -f $SPOOL_FILE $RESULT_FILE $SQL_FILE ${DBA_OUTPUT:-""}
fi

# Check, if there are no autoextend files.
if [ "$FILES_FOUND" -eq 0 ]
then
  print_msg W2 "No data files with autoextend option were found."
  print_msg .  "In this case, this monitor makes no sense."
  exit 2
fi

exit $RETVAL
