#!/usr/bin/sh
# OPC_WHAT_STRING="@(#)HP OpenView IT/Operations A.04.12 (01/08/99)"
################################################################################
#
# File:         opc_sec_verify.sh
# Description:  Script to verify the required DCE security setup for ITO.
# Language:     Bourne Shell 
# Package:      HP OpenView OperationsCenter
#
# (c) Copyright Hewlett-Packard Co. 1993 - 1999
#
################################################################################

# set -x

BIN_DIR=/usr/bin
OS_DIR=/usr
OPT_DIR=/opt
VAR_DIR=/var/opt
ETC_DIR=/etc/opt

# localization
LANG_C=C
LANG_SJIS=ja_JP.SJIS
LANG_EUC=ja_JP.eucJP

# for agents we may need a different root directory e.g. in a cluster
# environment else it is the root directory
#
ROOT_DIR=${ROOT_DIR:-""}

# defined in every script - the version of this script
INST_OPCVERS=A.04.12

# prerequisites:
#---------------
#
# the paths
#
#   OPT_DIR	e.g. /opt
#   VAR_DIR	e.g. /var/opt
#   ETC_DIR	e.g. /etc/opt
#
# must be defined before using this module
#
# Naming Scheme
# <prog>_<name>_DIR	1 directory e.g. OV_<name>_DIR, OPC_<name>_DIR
# <name>_PATH		several directories delimited by ":"
#
TMP_DIR=${ROOT_DIR}/tmp/opc_tmp

#-------------------------------------
# generic paths
#-------------------------------------
# /opt/OV
OV_OPT_DIR=${OV_OPT_DIR:-${ROOT_DIR}${OPT_DIR}/OV}
OPC_OPT_DIR=${OPC_OPT_DIR:-${OV_OPT_DIR}}
# /var/opt/OV
OV_VAR_DIR=${OV_VAR_DIR:-${ROOT_DIR}${VAR_DIR}/OV}
OV_VAR_SH_DIR=${OV_VAR_SH_DIR:-${OV_VAR_DIR}/share}
OPC_VAR_DIR=${OPC_VAR_DIR:-${OV_VAR_DIR}}
# /etc/opt/OV
OV_ETC_DIR=${OV_ETC_DIR:-${ROOT_DIR}${ETC_DIR}/OV}
OV_ETC_SH_DIR=${OV_ETC_SH_DIR:-${OV_ETC_DIR}/share}
OPC_ETC_DIR=${OPC_ETC_DIR:-${OV_ETC_DIR}/share}

OV_NEWCFG_DIR=${OV_NEWCFG_DIR:-${OV_OPT_DIR}/newconfig}
OV_OLDCFG_DIR=${OV_OLDCFG_DIR:-${OV_OPT_DIR}/old}

# WWW
OV_WWW_DIR=$OV_OPT_DIR/www

# /opt/OV
OV_OPT10_DIR=${OV_OPT10_DIR:-${ROOT_DIR}${OPT10_DIR}/OV}
# /var/opt/OV
OV_VAR10_DIR=${OV_VAR10_DIR:-${ROOT_DIR}${VAR10_DIR}/OV}
OV_VAR10_SH_DIR=${OV_VAR10_SH_DIR:-${OV_VAR10_DIR}/share}
# /etc/opt/OV
OV_ETC10_DIR=${OV_ETC10_DIR:-${ROOT_DIR}${ETC10_DIR}/OV}
OV_ETC10_SH_DIR=${OV_ETC10_SH_DIR:-${OV_ETC10_DIR}/share}

# where to find the minimal SD binaries
OPC_SD_DIR=${OPC_SD_DIR:-/usr/sbin}

#-------------------------------------
# OpenView paths
#-------------------------------------
OV_BIN_DIR=${OV_BIN_DIR:-${OV_OPT_DIR}/bin}
OV_TMP_DIR=${OV_TMP_DIR:-${OV_VAR_DIR}/tmp}
OV_LOG_DIR=${OV_LOG_DIR:-${OV_VAR_DIR}/log}
OV_CFG_DIR=${OV_CFG_DIR:-${OV_ETC_DIR}/share/conf}
OV_ALT_CFG_DIR=${OV_ALT_CFG_DIR:-${OV_ETC_DIR}/share/\#conf}

#-------------------------------------
# ITO Server Paths
#-------------------------------------
# /usr/OV/bin/OpC
OPCSVR_BIN_DIR=${OPCSVR_BIN_DIR:-${OPC_OPT_DIR}/bin/OpC}
# /usr/OV/bin/OpC/install
OPCSVR_INST_DIR=${OPCSVR_INST_DIR:-${OPCSVR_BIN_DIR}/install}
# /opt/OV/lib
OPCSVR_LIB_DIR=${OPCSVR_LIB_DIR:-${OPC_OPT_DIR}/lib}
# /opt/OV/nls
OPCSVR_NLS_DIR=${OPCSVR_NLS_DIR:-${OPCSVR_LIB_DIR}/nls}
# /opt/OV/include
OPCSVR_INC_DIR=${OPCSVR_INC_DIR:-${OPC_OPT_DIR}/include}
# /usr/OV/bin/OpC/utils
OPCSVR_UTIL_DIR=${OPCSVR_UTIL_DIR:-${OPCSVR_BIN_DIR}/utils}
# /usr/OV/bin/OpC/agtinstall
OPCSVR_AGTINST_DIR=${OPCSVR_AGTINST_DIR:-${OPCSVR_BIN_DIR}/agtinstall}
# /etc/opt/OV/share/lrf
OPCSVR_OPC_DIR=${OPCSVR_OPC_DIR:-${OPC_OPT_DIR}/OpC}
OPCSVR_NEWCFG_DIR=${OPCSVR_NEWCFG_DIR:-${OV_NEWCFG_DIR}/OpC}
OPCSVR_OLDCFG_DIR=${OPCSVR_OLDCFG_DIR:-${OV_OLDCFG_DIR}/OpC}
OPCSVR_HELP_DIR=${OPCSVR_HELP_DIR:-${OV_VAR_DIR}/share/help}

# /etc/opt/OV directories
OPCSVR_LRF_DIR=${OPCSVR_LRF_DIR:-${OPC_ETC_DIR}/lrf}
OPCSVR_REG_DIR=${OPCSVR_REG_DIR:-${OPC_ETC_DIR}/registration}

# /var/opt/OV/share/databases
OPCSVR_DB_DIR=${OPCSVR_DB_DIR:-${OPC_VAR_DIR}/share/databases/OpC}
# /var/opt/OV/share/databases/OpC/mgd_node/vendor
OPCSVR_VEND_DIR=${OPCSVR_DB_DIR}/mgd_node/vendor
# /var/opt/OV/share/databases/OpC/mgd_node/customer
OPCSVR_CUST_DIR=${OPCSVR_DB_DIR}/mgd_node/customer

# dynamic paths
OPCSVR_TMP_DIR=${OPCSVR_TMP_DIR:-${OPC_VAR_DIR}/share/tmp/OpC}
OPCSVR_LOG_DIR=${OPCSVR_LOG_DIR:-${OPC_VAR_DIR}/log/OpC}
OPCSVR_CFG_DIR=${OPCSVR_CFG_DIR:-${OPC_ETC_DIR}/conf/OpC}
OPCSVR_APPL_DIR=${OPCSVR_APPL_DIR:-${OPC_VAR_DIR}/share/tmp/OpC_appl}
OPCSVR_MAN_DIR=${OPCSVR_MAN_DIR:-${OPC_OPT_DIR}/man}

OPCSVR_INFO_F=${OPCSVR_INFO_F:-${OPCSVR_INST_DIR}/opcsvinfo}

# Config file needed by distributed GUI client
OPC_GUICLT_MSV_F=${OPC_GUICLT_MSV_F:-${OV_CFG_DIR}/opc_guiclt_msv}
OPC_GUICLT_TMP_MSV_F=${OPC_GUICLT_TMP_MSV_F:-/tmp/opc_guiclt_msv}

# DB config file
OVDB_CONFIG_FILE=${OVDB_CONFIG_FILE:-${OV_CFG_DIR}/ovdbconf}

#-------------------------------------
# ITO Agent Paths
#-------------------------------------
# /usr/OV/bin/OpC/s700
OPCAGT_BIN_DIR=${OPCAGT_BIN_DIR:-${OPC_OPT_DIR}/bin/OpC}
OPCAGT_LIB_DIR=${OPCAGT_LIB_DIR:-${OPC_OPT_DIR}/lib}
OPCAGT_NLS_DIR=${OPCAGT_NLS_DIR:-${OPCAGT_LIB_DIR}/nls}
OPCAGT_INC_DIR=${OPCAGT_INC_DIR:-${OPC_OPT_DIR}/include}
# /usr/OV/bin/OpC/s700/utils
OPCAGT_UTIL_DIR=${OPCAGT_UTIL_DIR:-${OPCAGT_BIN_DIR}/utils}
# /usr/OV/bin/OpC/s700/install
OPCAGT_INST_DIR=${OPCAGT_INST_DIR:-${OPCAGT_BIN_DIR}/install}

# /var/opt/OV/bin/OpC/monitor
OPCAGT_MON_DIR=${OPCAGT_MON_DIR:-${OPC_VAR_DIR}/bin/OpC/monitor}
# /opt/OV/bin/OpC/actions
OPCAGT_ACT_DIR=${OPCAGT_ACT_DIR:-${OPC_VAR_DIR}/bin/OpC/actions}
# /var/opt/OV/bin/OpC/cmds
OPCAGT_CMD_DIR=${OPCAGT_CMD_DIR:-${OPC_VAR_DIR}/bin/OpC/cmds}

# dynamic paths
OPCAGT_LOG_DIR=${OPCAGT_LOG_DIR:-${OPC_VAR_DIR}/log/OpC}
OPCAGT_CFG_DIR=${OPCAGT_CFG_DIR:-${OPC_VAR_DIR}/conf/OpC}
OPCAGT_TMP_DIR=${OPCAGT_TMP_DIR:-${OPC_VAR_DIR}/tmp/OpC}
OPCAGT_TMPBIN_DIR=${OPCAGT_TMPBIN_DIR:-${OPCAGT_TMP_DIR}/bin}
OPCAGT_TMPCFG_DIR=${OPCAGT_TMPCFG_DIR:-${OPCAGT_TMP_DIR}/conf}
OPCAGT_NEWCFG_DIR=${OPCAGT_NEWCFG_DIR:-${OV_NEWCFG_DIR}/OpC}
OPCAGT_OLDCFG_DIR=${OPCAGT_OLDCFG_DIR:-${OV_OLDCFG_DIR}/OpC}

# info files
OPCAGT_NDINFO_F=${OPCAGT_NDINFO_F:-${OPCAGT_CFG_DIR}/nodeinfo}
OPCAGT_INFO_F=${OPCAGT_INFO_F:-${OPCAGT_INST_DIR}/opcinfo}
OPCAGT_NETLS_F=${OPCAGT_NETLS_F:-${OPCAGT_CFG_DIR}/opcnetls}
OPCAGT_MGRCONF_F=${OPCAGT_MGRCONF_F:-${OPCAGT_CFG_DIR}/mgrconf}
OPCAGT_UPD_F=${OPCAGT_UPD_F:-${OPCAGT_TMP_DIR}/update}
OPCAGT_CLIENT_F=${OPCAGT_CLIENT_F:-${OPCAGT_INST_DIR}/cfg.clients}
OPCAGT_MGMTSV_F=${OPCAGT_MGMTSV_F:-${TMP_DIR}/mgmt_sv.dat}
#
# commands
#
OPCMON=${OPCAGT_BIN_DIR}/opcmon
OPCMSG=${OPCAGT_BIN_DIR}/opcmsg

VUE_DIR=/etc/vue
VUE_USR_DIR=/usr/vue
VUE_ETC_DIR=/etc/vue
VUE_VAR_DIR=/var/vue
CDE_USR_DIR=/usr/dt
CDE_ETC_DIR=/etc/dt
CDE_VAR_DIR=/var/dt

#-------------------------------------
# paths
#-------------------------------------
SYSTEM_PATH=${SYSTEM_PATH:-"/bin:/usr/bin:/sbin:/usr/sbin:/etc:/usr/etc:/usr/ucb"}
OV_PATH=${OV_BIN_DIR}
OPCSVR_PATH=${OPCSVR_BIN_DIR}:${OPCSVR_AGTINST_DIR}:${OPCSVR_INST_DIR}
OPCAGT_PATH=${OPCAGT_BIN_DIR}:${OPCAGT_INST_DIR}:${OPCAGT_MON_DIR}:${OPCAGT_ACT_DIR}:${OPCAGT_CMD_DIR}

# program name
APPLNAME=ITO
LONG_APPLNAME="HP OpenView IT/Operations 4.0"

# common umask value
umask 022

# print_msg() updates EXIT_CODE automatically using
# this utility function for script exit value.
# it keeps the worst exit code in ${EXIT_CODE}
#--------------------------------------------------
SUCCESS=0
FAILURE=1
WARNING=2

EXIT_CODE=${SUCCESS}

eval_exit()
{
  exit_new=$1

  if [ ${exit_new} -eq ${FAILURE} ]
  then
    EXIT_CODE=${FAILURE}
  elif [ ${exit_new} -eq ${WARNING} -a ${EXIT_CODE} -eq ${SUCCESS} ]
  then
    EXIT_CODE=${WARNING}
  fi

  return 0
}

#-------------------------------
#
# print_msg: print message to stderr/stdout and optional to logfile
#
#-------------------------------
# usage: print_msg <mode> <messages> ...
#
# <mode>: ( <type> | <fd> | <type><fd> | "." )
#
#        <type>:     use this type with <fd> = 2
#        <fd>:       use this fd with <type> = I
#        <type><fd>: use type and fd
#        ".":        use empty prefix and fd of last call
#
# <type>: ( "I" | "E" | "W" | "N" | "U" )
#          defines message prefix:
#          I)nfo:    "         "
#          E)rror:   "ERROR:   "
#          W)arning: "WARNING: "
#          N)ote:    "NOTE:    "
#          U)sage:   "USAGE:   "
#
# <fd>: file descriptor for output (2 = stderr, 1 = stdout)
#
# <messages>: the first message argument will be printed with
#             the selected prefix. Optionally following message
#             arguments will be printed in seperate lines with
#             an empty prefix.
# 
#-------------------------------
# examples:
#
# 1.) print_msg W "Something went wrong"
#
#     output to stderr: WARNING: Something went wrong
#
# 2.) print_msg N1 "Be aware that"
#     print_msg .  "something has happened"
#
#     output to stdout: NOTE:    Be aware that
#                     :          something has happened
#
# 3.) print_msg E2 "this is" "an" "error"
#
#     output to stderr: ERROR:   this is
#                     :          an
#                     :          error
#
#-------------------------------
# logging:
#
# default: logging is disabled
# enable logging with:  enable_logging <logfile>
# disable logging with: disable_logging
#
# if logging is enabled, all output of print_msg
# is also printed into the specified logfile.
#    
#--------------------------------
# exit codes:
#
# print_msg calls the shlib_misc:eval_exit() function
# with an appropriate parameter 
# in order to keep the EXIT_CODE up to date
#
#--------------------------------
MSG_LOGFILE=""

MSG_INFO="I"
MSG_ERROR="E"
MSG_WARN="W"
MSG_NOTE="N"
MSG_USAGE="U"
MSG_QUERY="Q"
MSG_LOG="L"    # goes to log only if enabled
MSG_SAME="."

#
# prefixes must have length 9
#                 123456789
      MSG_PREFIX="         "
 MSG_PREFIX_INFO="         "
MSG_PREFIX_ERROR="ERROR:   "
 MSG_PREFIX_WARN="WARNING: "
 MSG_PREFIX_NOTE="NOTE:    "
MSG_PREFIX_USAGE="USAGE:   "
MSG_PREFIX_QUERY="-------> "
MSG_PREFIX_LOG="LOG:     "

MSG_FD=2

print_msg()
{
  mode="$1"; shift

  if [ "$mode" = "$MSG_SAME" ]
  then
    type="$MSG_SAME"
  else
    type=`echo "$mode" | sed 's/^[^A-Z]*\([A-Z]\).*$/\1/'`
    MSG_FD=`echo $mode"_2" | sed 's/^[^0-9]*\([0-9][0-9]*\).*$/\1/'`
  fi

  prefix="$MSG_PREFIX"

  # set prefix
  case "$type" in
    $MSG_INFO)
      prefix="$MSG_PREFIX_INFO"
      ;;
    $MSG_ERROR)
      prefix="$MSG_PREFIX_ERROR"
      ;;
    $MSG_WARN)
      prefix="$MSG_PREFIX_WARN"
      ;;
    $MSG_NOTE)
      prefix="$MSG_PREFIX_NOTE"
      ;;
    $MSG_USAGE)
      prefix="$MSG_PREFIX_USAGE"
      ;;
    $MSG_QUERY)
      prefix="$MSG_PREFIX_QUERY"
      ;;
    $MSG_LOG)
      prefix="$MSG_PREFIX_LOG"
      ;;
    $MSG_SAME)
      prefix="$MSG_PREFIX"
      ;;
  esac

  for msg in "$@" ; do
    # concatenate with prefix
    msg="$prefix$msg"
    prefix="         "
  
    if [ "$type" != "$MSG_LOG" ]
    then
      # output message
      echo "$msg" >&$MSG_FD
    fi
  
    # if logging is enabled log the message
    if [ -n "$MSG_LOGFILE" ]
    then
      echo "$msg" >> ${MSG_LOGFILE}
    fi

  done

  # set EXIT_CODE
  case "$type" in
    $MSG_ERROR)
      eval_exit ${FAILURE}
      ;;
    $MSG_WARN)
      eval_exit ${WARNING}
      ;;
  esac
}

enable_logging()
{
  MSG_LOGFILE="$1"
}

disable_logging()
{
  MSG_LOGFILE=""
}

# variable defines
#-----------------
PASSWD_F=/etc/passwd
PASSWD_F_BAK=/etc/passwd.opc_bak
GROUP_F=/etc/group
GROUP_F_BAK=/etc/group.opc_bak

# functions in this module
#-------------------------
#

get_uid()
{
  id -u
}

is_user()
{
  uid=$1

  if [ `get_uid` -eq ${uid} ]
  then
    return 0
  else
    return 1
  fi
}

is_root()
{
  is_user 0
}

NEWCONF_DIR=${OPCSVR_NEWCFG_DIR:-${OPCAGT_NEWCFG_DIR}}
OLDCONF_DIR=${OPCSVR_OLDCFG_DIR:-${OPCAGT_OLDCFG_DIR}}

# get long hostname

get_hostname()
{
  _node=`uname -n 2>/dev/null`

  _host=`hostname_s2l $_node`

  unset _node
  echo "$_host"
  return 0
}

hostname_s2l()
{
  _node=$1
  _host=$_node

  # get full system name
  if [ -z "`echo $_node|sed -e 's/[^.]//g'`" ]
  then
    # hostname is only short; call nslookup to get fully
    # qualified hostname
    _host=`nslookup $_node 2>/dev/null |grep "^Name:[\t ].*" |awk '{print $2}'`
    if [ -z "$_host" ]
    then
      # unexpected problems calling nslookup
      echo "$_node"
      return 1
    else
      if [ "$_host" = "$_node" ]
      then
         # WARNING !!! Long hostname for Management Server not available
         echo "$_node"
         return 2
      fi
    fi
  fi
  # else uname delivers already long hostname
 
  unset _node
  echo "$_host"
  return 0
}

hostname_l2s()
{
  _host=$1

  echo $_host|cut -d . -f 1
  unset _host
}

# --------------------------------------------------------------
#
# These are the default values used for configuration of the ITO processes
# in the DCE registry.
#

REGISTRY_BASE=/.:/sec/principal
REGISTRY_BASE_AGT=opc/opc-agt
REGISTRY_BASE_MGR=opc/opc-mgr

OPC_GROUP_AGT=opc-agt
OPC_GROUP_MGR=opc-mgr

OPC_ORG=opc

DEFAULT_PASSWD=opc
OPC_AGT_ADM=opc-agt-adm
OPC_AGT_ADM_PERMS=rcDmau
OPC_MGR_KEY_FILE=${OPCSVR_CFG_DIR}/mgmt_sv/opckey

OPC_PWD_CMD=${OPCSVR_INST_DIR}/opcpwd

# --------------------------------------------------------------
#   $1 = Exit code from the rgy_edit command
#   $2 = Temporary logfile containing the output of rgy_edit
#   $3 = Message texts from rgy_edit which do not cause an error
#   $4 = Error message to report
# --------------------------------------------------------------
eval_rgy_edit()
{
  exit_code=$1
  logfile=$2
  scan_for=$3
  msg_text=$4

  if [ ${exit_code} -ne 0 ]
  then
    grep -q "${scan_for}" ${logfile}
    if [ $? -ne 0 -o -z "${scan_for}" ]
    then
      print_msg E "${msg_text}"
      cat ${logfile}
      rm ${logfile}
      exit 1
    fi
  fi
}

# --------------------------------------------------------------
#   $1 = Exit code from the acl_edit command
#   $2 = Temporary logfile containing the output of acl_edit
#   $3 = Error message to report
# --------------------------------------------------------------
eval_acl_edit()
{
  exit_code=$1
  logfile=$2
  msg_text=$3

  if [ ${exit_code} -ne 0 ]
  then
    print_msg E "${msg_text}"
    cat ${logfile}

    grep -q "unauthenticated" ${logfile}
    if [ $? -eq 0 ]
    then
      print_msg . "An unauthenticated binding can happen if you logged on to"
      print_msg . "DCE and changed your identity using su(1) afterwards."
      print_msg . "You must first become root and then log on to DCE."
    fi

    grep -q "acl object not found" ${logfile}
    if [ $? -eq 0 ]
    then
      print_msg . "One possible reason for this error is if the DCE cdsd is not"
      print_msg . "running."
      print_msg . "Check and restart all DCE daemons if necessary using the"
      print_msg . "dce_config program."
    fi

    rm ${logfile}
    exit 1
  fi
}

PATH="${PATH}:${SYSTEM_PATH}:${OPCSVR_PATH}:${OPCAGT_PATH}"
export PATH

###############################################################################
#
# This script verifies DCE registry entries for ITO processes. It can be
# used on the management server only and is supposed to be integrated into the
# application bank.
#
# Usage opc_sec_verify.sh [ -help ] 
#       See function print_usage().
#
# The <node> parameters can be used to check entries for the specified nodes.
# If no node is specified the local managed node will be assumed.
#
# Always the information relevant to the management server will be displayed.
#
# We do not require a DCE login context - usually the DCE registry allows
# unauthenticated read access.
#
###############################################################################

#
# Local functions
#---------------------------------------------------------------------------

print_usage()
{
  print_msg U "$0 [-help]"
  print_msg . "$0 [<node> ...]"
}

verify_node_princ()
{
  princ=$1
  exp_grp=$2
  exp_org=$3
  ALL_OK=1

  echo ""
  
  ${RGY_EDIT} -p -v ${princ} >${TMP_LOG_FILE} 2>&1
  PRC_P_EXISTS=`awk '{ if($1 == prc) print $0}' prc=${princ} ${TMP_LOG_FILE}`
  eval_rgy_edit $? ${TMP_LOG_FILE} '' \
       "Cannot retrieve entries for ${princ} from DCE security service."

  ${RGY_EDIT} -a -v ${princ} -f >${TMP_LOG_FILE} 2>&1
  PRC_A_EXISTS=`awk '{ if($1 == prc) print $0}' prc=${princ} ${TMP_LOG_FILE}`
  ACCT_VALID=`awk '{ if($1 == "Account" && $2 == "is:") print $3}' \
             ${TMP_LOG_FILE}`
  PASSWD_VALID=`awk '{ if($1 == "password" && $2 == "is:") print $3}' \
             ${TMP_LOG_FILE}`
  REAL_PGO=`awk '{if(match($0, "\[.*\]")) print $0}' ${TMP_LOG_FILE}`
  eval_rgy_edit $? ${TMP_LOG_FILE} '' \
       "Cannot retrieve entries for ${princ} from DCE security service."

  if [ -z "${PRC_P_EXISTS}" ]
  then
    print_msg E "The principal ${princ} does not exist at all."
    return 1
  fi

  if [ -z "${PRC_A_EXISTS}" ]
  then
    if [ -z "${PRC_P_EXISTS}" ]
    then
      print_msg E "The principal ${princ} has an account but is not known as"
      print_msg . "principal. This is an illegal situation within DCE!"
    else
      print_msg E "The principal ${princ} does not have an account but exists"
      print_msg . "as principal."
    fi
    return 1
  fi

  if [ "${ACCT_VALID}" != "valid" ]
  then
    print_msg E "The account of ${princ} is not valid."
    ALL_OK=0
  fi

  if [ "${PASSWD_VALID}" != "valid," ]
  then
    print_msg E "The password of ${princ} is not valid."
    ALL_OK=0
  fi

  PGO=`echo ${REAL_PGO} | sed -e"s/^.*\[//" -e "s/\].*$//"`
  REAL_GRP=`echo ${PGO} | awk '{print $1}'`
  REAL_ORG=`echo ${PGO} | awk '{print $2}'`

  if [ "${REAL_GRP}" != "${exp_grp}" ]
  then
    print_msg W "The account of ${princ} is not in the ITO default group."
    print_msg . "Instead it is in group ${REAL_GRP} - but this is not a"
    print_msg . "critical condition."
    ALL_OK=0
  fi

  if [ "${REAL_ORG}" != "${exp_org}" ]
  then
    print_msg W "The account ${princ} is not in the ITO default organization."
    print_msg . "Instead it is in organization ${REAL_ORG} - but this is not a"
    print_msg . "critical condition."
    ALL_OK=0
  fi

  if [ ${ALL_OK} -eq 1 ]
  then
    print_msg I "The principal ${princ} exists as:\n\t\t${PRC_P_EXISTS}."
    print_msg . "Both account and password are valid."
    print_msg . "Everything seems to be OK."
  else
    print_msg W "Some inconsistencies were detected. Here is a complete report"
    print_msg . "about this principal:"
    print_msg . "-------------------------------------------------------------"
    ${RGY_EDIT} -p -v ${princ} -f
    ${RGY_EDIT} -a -v ${princ} -f
    print_msg . "-------------------------------------------------------------"
  fi

  return 0
}

#
# Script body
#---------------------------------------------------------------------------

TMP_LOG_FILE=/tmp/opc_sec.log
E_CODE=0
SERVER=0

RGY_EDIT=/opt/dcelocal/bin/rgy_edit

#
# Check options.
#---------------------------------------------------------------------------

if [ "$1" = "-h" -o "$1" = "-help" ]
then
  print_usage
  exit 0
fi

OPT_IS_VALID=1

while [ ${OPT_IS_VALID} -eq 1 ]
do
  OPT_IS_VALID=0

  if [ "$1" = "-s" -o "$1" = "-server" ]
  then
    SERVER=1
    OPT_IS_VALID=1
    unset NODE_LIST
    shift
  fi
done

LOCAL_NODE=`get_hostname`

if [ ${SERVER} -eq 0 ]
then
  if [ $# -eq 0 ]
  then
    NODE_LIST=${LOCAL_NODE}
  else
    NODE_LIST=$*
  fi
fi

echo ""
print_msg I "ITO verification utility for DCE security setup"
print_msg . "==============================================="
echo ""

is_root
if [ $? -ne 0 ]
then
  print_msg W "You are not root - certain operations like accessing the ITO"
  print_msg . "key tab files and ACL entries will fail due to missing"
  print_msg . "permissions."
fi

#
# First check whether we can access the DCE services. It makes no sense to
# check whether the processes are running - possibly the DCE server system is
# not ITO management server at the same time.
#---------------------------------------------------------------------------

if [ ! -x ${RGY_EDIT} ]
then
  print_msg E "No program ${RGY_EDIT} available. Check your DCE installation."
  exit 1
fi

print_msg I "***** Trying to contact the DCE security service ... *****"
echo ""

echo "site\nview" | ${RGY_EDIT} >${TMP_LOG_FILE} 2>&1
MY_CELL=`grep "Current site" ${TMP_LOG_FILE} | awk '{ print $7}'`
IS_NOT_RUNNING=`grep "Not registered" ${TMP_LOG_FILE} | wc -l`
eval_rgy_edit $? ${TMP_LOG_FILE} 'not authenticated' \
                "Cannot retrieve cell local name."

if [ -z "${MY_CELL}" ]
then
  print_msg E "The local host doesn't seem to be configured as member of a DCE"
  print_msg . "cell. This is the basic prerequisite to run ITO using"
  print_msg . "authenticated RPC."
  print_msg . "Use the 'dce_config' utility to configure the local host as"
  print_msg . "cell member."
  exit 1
else

  #
  # If the contact succeeded the variable ${MY_CELL} contains the cell name as
  # well as the security server system. So we need to split this string.
  #

  if [ ${IS_NOT_RUNNING} -eq 1 ]
  then
    print_msg I "The local host is member of the DCE cell '${MY_CELL}'."
  else
    echo ${MY_CELL} | \
      awk '{ num_cmps = split($0, path, "/"); \
             printf "\t\tThe local host is member of the DCE cell \""; \
             for(i = 1; i < num_cmps && path[i] != "subsys"; i++) \
             { \
               if(i > 1) \
               { \
                 printf "/"\
               } \
               printf "%s", path[i]; \
             } \
             printf "\"\n\t\tSecurity server is \"%s\"\n", path[num_cmps]; \
           }'
  fi
fi

echo ""

if [ ${IS_NOT_RUNNING} -eq 1 ]
then
  print_msg W "The DCE security server is supposed to be registered at the"
  print_msg . "local rpcd/dced but is not. This means that the local system is"
  print_msg . "configured as DCE server system but the security server process"
  print_msg . "is not running."
  print_msg . "Use the 'dce_config' utility to start them."
  exit 1
else
  print_msg I "Contacting the DCE security service succeeded."
fi

echo ""

#
# Check the groups and organization used by the OpC processes.
#---------------------------------------------------------------------------

print_msg I "***** Checking groups and organizations ... *****"
echo ""

${RGY_EDIT} -o -v >${TMP_LOG_FILE} 2>&1
OPC_ORG_EXISTS=`awk '{ if($1 == org) print $0}' org=${OPC_ORG} ${TMP_LOG_FILE}`
eval_rgy_edit $? ${TMP_LOG_FILE} 'not authenticated' \
              "Cannot retrieve organization information from registry."

if [ -z "${OPC_ORG_EXISTS}" ]
then
  print_msg W "The ITO default organization ${OPC_ORG} does not exist."
else
  print_msg I "The ITO default organization exists as:\n\t\t${OPC_ORG_EXISTS}."

  echo "policy ${OPC_ORG}\n" | ${RGY_EDIT} >${TMP_LOG_FILE} 2>&1
  ORG_POL=`awk 'BEGIN { eff_pol = 0; acct=0; pwd=0 } \
       { if(eff_pol == 0) \
         { \
           if($1 == "Effective" && $2 == "Policy:") \
           { \
             eff_pol = 1 \
           } \
         } \
         else \
         { \
           if($1 == "Account" && $2 == "lifespan:" && $3 != "forever") \
           { \
             acct=1
           } \
           if(($1 == "Password") && \
             (($2 == "lifespan:" && $3 != "forever") || \
              ($2 == "expiration" && $4 != "none"))) \
           { \
             pwd=1
           } \
         } \
       } \
       END { printf "a=%d p=%d", acct, pwd }' ${TMP_LOG_FILE}`
  eval_rgy_edit $? ${TMP_LOG_FILE} 'not authenticated' \
                "Cannot retrieve policy information from registry."

  echo ${ORG_POL} | grep -q "a=1"
  if [ $? -eq 0 ]
  then
    print_msg W "Account lifetime policy for this organization is limited."
    print_msg . "This is potentially dangerous because there are no automatic"
    print_msg . "account refreshment mechanisms."
    echo ""
  fi
  echo ${ORG_POL} | grep -q "p=1"
  if [ $? -eq 0 ]
  then
    print_msg W "Password lifetime policy for this organization is limited."
    print_msg . "The ITO agent processes will attempt to generate a new"
    print_msg . "password when necessary. This requires a correct account of"
    print_msg . "the ITO agent administrative principal opc-agt-adm."
    print_msg . "If the ITO management server accounts are member of the"
    print_msg . "default organization ${OPC_ORG} their passwords will expire,"
    print_msg . "too, and need to be refreshed manually e.g. using the ITO"
    print_msg . "utility opc_sec_register -s."
    echo ""
  fi
fi

${RGY_EDIT} -g -v >${TMP_LOG_FILE} 2>&1
AGT_GRP_EXISTS=`awk '{ if($1 == grp) print $0}' grp=${OPC_GROUP_AGT} \
               ${TMP_LOG_FILE}`
MGR_GRP_EXISTS=`awk '{ if($1 == grp) print $0}' grp=${OPC_GROUP_MGR} \
               ${TMP_LOG_FILE}`
eval_rgy_edit $? ${TMP_LOG_FILE} 'not authenticated' \
              "Cannot retrieve group information from registry."

if [ -z "${AGT_GRP_EXISTS}" ]
then
  print_msg W "The ITO default agent group ${OPC_GROUP_AGT} does not exist."
else
  print_msg I "The ITO default agent group exists as:\n\t\t${AGT_GRP_EXISTS}."
fi

if [ -z "${MGR_GRP_EXISTS}" ]
then
  print_msg W "The ITO default manager group ${OPC_GROUP_MGR} does not exist."
else
  print_msg I "The ITO default manager group exists as:\n\t\t${MGR_GRP_EXISTS}."
fi

#
# Check the principals for all nodes used by the OpC processes. First for the
# local management server.
#---------------------------------------------------------------------------

if [ ${SERVER} -eq 1 ]
then
  echo ""
  print_msg I "***** Checking management server principal ... *****"

  OPC_MGR=${REGISTRY_BASE_MGR}/${LOCAL_NODE}
  verify_node_princ ${OPC_MGR} ${OPC_GROUP_MGR} ${OPC_ORG}
fi

#
# Then deal with the managed nodes passed as arguments - if there aren't any
# take the local system.
#---------------------------------------------------------------------------

echo ""
print_msg I "***** Checking managed node principals ... *****"

for NODE in ${NODE_LIST}
do
  LONG_NODE=`hostname_s2l ${NODE}`
  OPC_AGT=${REGISTRY_BASE_AGT}/${LONG_NODE}

  verify_node_princ ${OPC_AGT} ${OPC_GROUP_AGT} ${OPC_ORG}

  echo ""
  ACL_PERMS=`acl_edit ${REGISTRY_BASE}/${OPC_AGT} -l | grep ${OPC_AGT_ADM}`
  if [ -z "${ACL_PERMS}" ]
  then
    print_msg W "No ACL entry found for ${OPC_AGT_ADM} and ${OPC_AGT}."
    print_msg . "Automatic password generation will not work for this node."
  else
    if [ "${ACL_PERMS}" != "user:${OPC_AGT_ADM}:rcD--mau-" ]
    then
      print_msg W "The ACL entry found for ${OPC_AGT_ADM} and ${OPC_AGT}"
      print_msg . "is incorrect: ${ACL_PERMS}."
      print_msg . "Automatic password generation will not work for this node."
    fi
  fi
done

#
# Finally inspect the opc-agt-adm account.
#---------------------------------------------------------------------------

verify_node_princ ${OPC_AGT_ADM} ${OPC_GROUP_AGT} "none"

echo ""
print_msg I "To verify all details of a principal and account run the following"
print_msg . "commands:"
print_msg . "\t# rgy_edit -p -v <prinicipal> -f"
print_msg . "\t# rgy_edit -a -v <prinicipal> -f"

#
# Verify the keytab files
#---------------------------------------------------------------------------

echo ""
print_msg I "***** Checking key tab files ... *****"
echo ""

if [ ${SERVER} -eq 1 ]
then
  if [ ! -f ${OPC_MGR_KEY_FILE} ]
  then
    print_msg E "The management server key tab file ${OPC_MGR_KEY_FILE} does"
    print_msg . "not exist."
  else
    echo "ktl -f ${OPC_MGR_KEY_FILE}" | ${RGY_EDIT} >${TMP_LOG_FILE} 2>&1
    NUM_MGR_KEYS=`grep ${OPC_MGR} ${TMP_LOG_FILE} | wc -l`
    eval_rgy_edit $? ${TMP_LOG_FILE} '' "Cannot verify key tab file."
    if [ ${NUM_MGR_KEYS} -eq 0 ]
    then
      print_msg E "The management server key tab file ${OPC_MGR_KEY_FILE} does"
      print_msg . "not contain a password for ${OPC_MGR}."
    else
      print_msg I "The management server key tab file ${OPC_MGR_KEY_FILE}"
      print_msg . "contains ${NUM_MGR_KEYS} keys for ${OPC_MGR}."
    fi
  fi
fi

if [ "${NODE_LIST}" = "${LOCAL_NODE}" ]
then
  AGT_KEY_FILE=${OPCAGT_CFG_DIR}/opckey
  OPC_AGT=${REGISTRY_BASE_AGT}/${LOCAL_NODE}

  if [ ! -f ${AGT_KEY_FILE} ]
  then
    print_msg E "The managed node key tab file ${AGT_KEY_FILE} does not exist."
  else
    echo "ktl -f ${AGT_KEY_FILE}" | ${RGY_EDIT} >${TMP_LOG_FILE} 2>&1
    NUM_AGT_KEYS=`grep ${OPC_AGT} ${TMP_LOG_FILE} | wc -l`
    eval_rgy_edit $? ${TMP_LOG_FILE} '' "Cannot verify key tab file."
    if [ ${NUM_AGT_KEYS} -eq 0 ]
    then
      print_msg E "The managed node key tab file ${AGT_KEY_FILE} does not"
      print_msg . "contain a password for ${OPC_AGT}."
    else
      print_msg I "The managed node key tab file ${AGT_KEY_FILE} contains"
      print_msg . "${NUM_AGT_KEYS} keys for ${OPC_AGT}."
    fi
  fi
fi

#
# Finally try to log on as all the principals.
#---------------------------------------------------------------------------

if [ ${SERVER} -eq 1 ]
then
  echo ""
  print_msg I "***** Logging on as ${OPC_MGR} ... *****"
  echo ""
  
  DCE_LOGIN_OUTP=`dce_login -k ${OPC_MGR_KEY_FILE} ${OPC_MGR} -e echo done`
  if [ "${DCE_LOGIN_OUTP}" = "done" ]
  then
    print_msg I "Logon succeeded."
  else
    print_msg W "Logon failed: ${DCE_LOGIN_OUTP}."
  fi
fi

if [ "${NODE_LIST}" = "${LOCAL_NODE}" ]
then
  AGT_KEY_FILE=${OPCAGT_CFG_DIR}/opckey
  OPC_AGT=${REGISTRY_BASE_AGT}/${LOCAL_NODE}

  echo ""
  print_msg I "***** Logging on as ${OPC_AGT} ... *****"
  echo ""

  DCE_LOGIN_OUTP=`dce_login -k ${AGT_KEY_FILE} ${OPC_AGT} -e echo done`
  if [ "${DCE_LOGIN_OUTP}" = "done" ]
  then
    print_msg I "Logon succeeded."
  else
    print_msg W "Logon failed: ${DCE_LOGIN_OUTP}."
  fi
fi

echo ""
print_msg I "***** All done. *****"

rm -f ${TMP_LOG_FILE}

exit ${E_CODE}

################################################################################
# End of file: opc_sec_verify.sh
################################################################################
