#!/serveur/dp/bin/perl -w

# 	$Id: MAIL_sur.pl,v 2.2 2001/09/27 07:13:57 p004184 Exp $	

# ---------------------------------------------------------------
#
# Surveillance des mailbox
#
# Le script indique :
# - les mailbox non touches depuis 6 mois
# - les mailbox dont le premier mail est vieux de 6 mois
# - les mailbox ayant plusieurs alias (dans la map aliases)
# - les mailbox n'ayant pas d'alias
# - les mailbox de plus de 3Mo
# - les rpertoires qui trainent
# - les mailbox avec un uid <> user ou gid <> mail
# - les mailbox avec des noms drivs (ex : toto1234ASDFG123_ driv de toto)
# 
# Il supprime directement :
# - les dt_index, toto, tata, titi, etc, vieux d'une semaine
# - les mailbox non touches depuis 6 mois et avec un nom contenant 1 point
#		(donc .Z, etc)
# - les mailbox non touches depuis 6 mois et vides (taille  zro)
# - les mailbox n'ayant pas d'alias et avec un nom se terminant par
#		.Z, .gz, .old
# - les mailbox ayant un alias en @renault.com
# - les mailbox ayant un alias sur une autre machine
# - des mailbox non touches depuis plus de 1 an
# - les rpertoires qui trainent depuis plus de 6 mois
# - les mailbox avec des noms drivs non touches depuis plus de 6 mois
#
# usage : MAIL_sur.pl <nom_template>
#
# Code retour :
#		0 : ok , 1 : erreur, 2 : alarme ITO
#
# exemple : MAIL_sur.pl MAIL_sur
#
# ---------------------------------------------------------------

use strict;
use File::Basename;
use Time::Local;
require "hostname.pl";
require "ctime.pl";
use lib dirname($0);
use dits_def;
use Fcntl ':flock';

my $log = "/var/tmp/opc_mailsur.log";

my $size_max = 3;		# seuil d'alerte en Mo

# tentatives pour reprer des fichiers  supprimer
my $noms_veroles = '\.(old|gz|Z)$';
my $poubelle = '^(.*_dt_index|toto\d*|titi\d*|tata\d*|test\d*)$';

# dlais pour suppression de certaines mailbox
my $semaine_derniere = time() - 7*24*60*60;		# si poubelle
my $vieux = time() - 180*24*60*60;				# si nom vrol ou driv ou rpertoire
my $tres_vieux = time() - 365*24*60*60;			# toutes mailbox

my %mois = ('Jan',1,'Feb',2,'Mar',3,'Apr',4,'May',5,'Jun',6,'Jul',7
			,'Aug',8,'Sep',9,'Oct',10,'Nov',11,'Dec',12);
my %fait;
my (%rep,%size);
my @aliases;
my @passwd;
my $host = hostname();
my $total = 0;

# ---------------------------------------------------------------
# 0: pas de trace, 1: traces
# ---------------------------------------------------------------
my $debug = Debug();

Trace "@ARGV" if($debug);

# ---------------------------------------------------------------
# stocke sortie dans fichier log
# ---------------------------------------------------------------
sub Log {
	my @param = @_;
	open(LOG,">>$log") || Trace "open $log : $!";
	flock(LOG,LOCK_EX) || Trace "lock $log : $!";
	print LOG @param;
	print @param;
	flock(LOG,LOCK_UN);
	close(LOG);
}

# ---------------------------------------------------------------
# recherche les aliases
# ---------------------------------------------------------------
sub lit_alias {
	@aliases = ypcat("-k aliases 2>&1");
	my $alias = "/etc/mail/aliases";
	$alias = "/etc/aliases" if(-f "/etc/aliases");
	if(open(ALIAS,$alias)) {
		while(<ALIAS>) {
			next if(/^\#|^\s*$/);
			push(@aliases,$_);
		}
		close(ALIAS);
	} else {
		Log "Erreur open $alias : $!\n";
		return 0;
	}
	return 1;
}

# ---------------------------------------------------------------
# recherche les comptes
# ---------------------------------------------------------------
sub lit_passwd {
	@passwd = ypcat("passwd 2>&1");
	my $passwd = "/etc/passwd";
	if(open(PASSWD,$passwd)) {
		while(<PASSWD>) {
			next if(/^\#|^\s*$/);
			push(@passwd,$_);
		}
		close(PASSWD);
	} else {
		Log "Erreur open $passwd : $!\n";
		return 0;
	}
	return 1;
}

# ---------------------------------------------------------------
# recherche les points de montage des comptes
# ---------------------------------------------------------------
sub mountpoint {
	my($compte) = @_;
	my $mountpoint = "";
	if($compte) {
		my @maps = ('auto.people','auto.adm','auto.support','auto.projet','auto.divers',
					'auto.bureautique',"auto.group.$host","auto.logiciel.$host.COMMUN",
					"auto.logiciel.$host.SGISVR4","auto.logiciel.$host.SUNSVR4","auto.bd",
					"auto.bdiao","auto.bdrobcad","auto.bdmt");
		for(@maps) {
			my(@temp) = ypmatch("$compte $_ 2> /dev/null");
			if(@temp) {
				chomp($mountpoint = $temp[0]);
				last;
			}
		}
	}
	return $mountpoint;
}

# ---------------------------------------------------------------
# supprime une mailbox, soit fichier, soit rpertoire
# ---------------------------------------------------------------
sub SupMailbox {
	my($file,$size) = @_;
	my $code = 1;
	if(-f $file) {
		Log "  --> suppression $file\n";
		if(! unlink($file)) {
			Log "  $file : erreur unlink : $!\n";
			$code = 0;
		} else {
			$total += $size;
		}
	} elsif(-d $file) {
		if(opendir(DIR,$file)) {
			my @dir = readdir(DIR);
			closedir(DIR);
			for(@dir) {
				next if(/^(\.|\.\.)$/);
				my($st_dev,$st_ino,$st_mode,$st_nlink,$st_uid,$st_gid,$st_rdev,$st_size,
				   $st_atime,$st_mtime,$st_ctime,$st_blksize,$st_blocks) = stat("$file/$_");
				if($st_mtime < $vieux) {
					$code = 0 if(!SupMailbox("$file/$_",$st_size));
				}
			}
			Log "  --> suppression $file\n";
			if(!rmdir($file)) {
				Log "  $file : erreur rmdir : $!\n";
				$code = 0;
			}
		} else {
			Log "  $file : erreur opendir : $!\n";
			$code = 0;
		}
	}
	return $code;
}

# ---------------------------------------------------------------
# renvoie la date du premier mail de la mailbox
# ---------------------------------------------------------------
sub PremierMail {
	my($file) = @_;
	my $time_access = time();
	if(open(MAIL,$file)) {
		my $line;
		while(defined($line = <MAIL>)) {
			if($line =~ /^From\s+\S+\s+(.+)$/) {
				my($day,$mois,$jour,$heure,$minute,$type,$annee) = split(/\s+|:/,$1);
				$time_access = timelocal(0,$minute,$heure,$jour,$mois{$mois}-1,$annee);
				last;
			}
		}
		close(MAIL);
	}
	return $time_access;
}

# ---------------------------------------------------------------
# Suppression des mails fdrs (en renault.com) ou dplacs
# sur un autre host.
# Suppression des vieilles mailbox en .old, etc.
# ---------------------------------------------------------------
sub VerifMailbox {
	my($file,@dir) = @_;
	my $code = 0;
	my $user = basename($file);
	return 0 if($fait{$user});
	$fait{$user} = 1;
	my $supprime = 0;
	my($st_dev,$st_ino,$st_mode,$st_nlink,$st_uid,$st_gid,$st_rdev,$st_size,
	   $st_atime,$st_mtime,$st_ctime,$st_blksize,$st_blocks) = stat($file);
	if(-d $file) {
		Log "  $user : c'est un rpertoire\n";
		if($st_mtime < $vieux) {
			Log "  $user : le rpertoire n'a pas boug depuis ",ctime($st_mtime);
			$supprime = SupMailbox($file,$st_size);
		}
		return 2;
	} elsif(! -f $file) {
		Log "  $user : n'est pas un fichier ni un rpertoire\n";
		return 2;
	}
	for(sort @dir) {
		next if(/^(\.|\.\.|:saved)$/);
		my $file2 = $_;
		if($user =~ /^$file2.*\d{2}/) { # 2 chiffres pour viter 'gdgre7 driv de gdg'
			Log "  $user : la mailbox est un driv de $file2\n";
			$code = 2 ;
			if($st_mtime < $vieux) {
				Log "  $user : la mailbox n'a pas bouge depuis ",ctime($st_mtime);
				$supprime = SupMailbox($file,$st_size);
			}
		}
	}
	my $real_uid = getpwuid($st_uid);
	if(!$supprime && $real_uid ne $user) {
		Log "  $user : la mailbox n'a pas le bon uid ($real_uid)\n";
		$code = 2;
	}
	my $real_gid = getgrgid($st_gid);
	if(!$supprime && $real_gid ne 'mail') {
		Log "  $user : la mailbox n'a pas le bon gid ($real_gid)\n";
		$code = 2;
	}
	if(!$supprime && $st_mtime < $tres_vieux) {
		Log "  $user : la mailbox n'a pas bouge depuis ",ctime($st_mtime);
		$code = 2;
		$supprime = SupMailbox($file,$st_size);
	}
	if(!$supprime && $st_mtime < $vieux) {
		Log "  $user : la mailbox n'a pas bouge depuis ",ctime($st_mtime);
		$code = 2;
		if($user =~ /$noms_veroles|\./) {
			Log "  $user : la mailbox n'a pas un beau nom\n";
			$supprime = SupMailbox($file,$st_size);
		} elsif($st_size==0) {
			Log "  $user : la mailbox est vide\n";
			$supprime = SupMailbox($file,$st_size);
		}
	}
	if(!$supprime && $user =~ /$poubelle/ && $st_mtime < $semaine_derniere) {
		Log "  $user : la mailbox n'a pas un beau nom\n";
		Log "  $user : ce fichier date du ",ctime($st_mtime);
		$supprime = SupMailbox($file,$st_size);
		$code = 2;
	}
	if(!$supprime) {
		my @alias_user;
		for(@aliases) {
			push(@alias_user,$1) if(/^$user[\s:]+(.*)$/);
		}
		if(@alias_user>1) {
			Log "  $user : plusieurs alias : @alias_user\n";
			$code = 2;
		} elsif(@alias_user==0) {
			Log "  $user : aucun alias\n";
			$code = 2;
			if($user =~ /$noms_veroles/) {
				Log "  $user : la mailbox n'a pas un beau nom\n";
				$supprime = SupMailbox($file,$st_size);
			}
		} elsif($alias_user[0] =~ /\@renault\.com(|,.+)$/i) {
			Log "  $user : migre en renault.com\n";
			$supprime = SupMailbox($file,$st_size);
			$code = 2;
		} elsif($alias_user[0] !~ /\@$host/i) {
			Log "  $user : aliase en $alias_user[0]\n";
			$supprime = SupMailbox($file,$st_size);
			$code = 2;
		}
	}
	if(!$supprime) {
		my @passwd_user;
		for(@passwd) {
			if(/^$user:/) {
				my($login,$passwd,$uid,$gid,$comment,$homedir,$shell) = split(/:/);
				push(@passwd_user,$homedir) if(grep(/^$homedir$/,@passwd_user)==0);
			}
		}
		if(@passwd_user>1) {
			Log "  $user : plusieurs comptes : @passwd_user\n";
			$code = 2;
		} elsif(@passwd_user==0) {
			Log "  $user : aucun compte\n";
			$code = 2;
		} else {
			my $mountpoint = mountpoint(basename($passwd_user[0]));
			if(!$mountpoint) {
				Log "  $user : compte sur disque local\n";
				#$code = 2;
			} else {
				my($hostname,$mountpoint) = split(/:/,$mountpoint);
				if($hostname ne $host) {
					Log "  $user : compte sur $hostname\n";
					$code = 2;
				} elsif(! -d $mountpoint) {
					Log "  $user : compte $passwd_user[0] inaccessible\n";
					$code = 2;
				}
			}
		}
	}
	if(!$supprime && $st_size>=$size_max) {
		$rep{$user} = dirname($file);
		$size{$user} = $st_size;
		$code = 2;
	}
	if(!$supprime && $code==2) {
		my $time_access = PremierMail($file);
		if($time_access < $vieux) {
			Log "  $user : le premier mail remonte au       ",ctime($time_access);
			$code = 2;
		}
	}
	return $code;
}

# ---------------------------------------------------------------
# Vrification d'un rpertoire
# ---------------------------------------------------------------
sub VerifDir {
	my($rep) = @_;
	my $code = 0;
	if(-d $rep && opendir(DIR,$rep)) {
		my @dir = readdir(DIR);
		closedir(DIR);
		for(sort @dir) {
			next if(/^(\.|\.\.|:saved)$/);
			my $file1 = $_;
			$code = 2 if(VerifMailbox("$rep/$file1",@dir)>0);
		}
	}
	return $code;
}

# ---------------------------------------------------------------
# Vrification des mails dans plusieurs rpertoires.
# ---------------------------------------------------------------
sub Traitement {
	Trace "Traitement" if($debug);
	my $code = 0;
	unlink $log if(-f $log);
	if(! lit_alias()) {
		Log "Abandon\n";
		$code = 2;
	}
	lit_passwd();
	my($mail_dir) = grep(/^$host/,ypcat("auto.mail"));
	if($mail_dir) {
		chomp($mail_dir);
		$mail_dir =~ s/^\S+://;
	}
	$size_max = $size_max*1024*1024;
	Log "\nAnomalies potentielles\n\n";
	if($mail_dir && $ENV{'MAIL'}) {
		if(dirname($ENV{'MAIL'}) ne $mail_dir) {
			Log "Incohrence auto.mail ($mail_dir) et \$MAIL (",dirname($ENV{'MAIL'}),")\n";
			$code = 2;
		}
	}
	if($mail_dir && -d $mail_dir) {
		$code = 2 if(VerifDir("$mail_dir")>0);
	} elsif($ENV{'MAIL'}) {
		$code = 2 if(VerifDir(dirname($ENV{'MAIL'}))>0);
	} else {
		$code = 2 if(VerifDir("/var/mail")>0);
		$code = 2 if(VerifDir("/var/spool/mail")>0);
		$code = 2 if(VerifDir("/usr/mail")>0);
	}
	if($code==0) {
		Log "  Aucune\n\n";
	}
	Log "\nFichiers mail depassant ", $size_max/(1024*1024),"Mo\n\n";
	if(! %size) {
		Log "  Aucun\n\n";
	} else {
		for(sort { $size{$b} <=> $size{$a} } keys %size) {	# tri sur size descending
			my $mes = sprintf "%10d  %s/%s\n",$size{$_},$rep{$_},$_;
			Log $mes;
		}
		Log "\n";
	}
	if($total>0) {
		my $mes = sprintf "\nTotal suppressions\n\n%10dKo\n\n",$total/1024;
		Log $mes;
	}
	return $code;
}

# ---------------------------------------------------------------
# lecture des parametres : process a surveiller
# ---------------------------------------------------------------
sub LectureParametres {
	Trace "LectureParametres" if($debug);
	return 0;
}

# ---------------------------------------------------------------
# traitement commun a tous les templates ITO
# ---------------------------------------------------------------

Trace "@ARGV" if($debug);

# recuperation du nom de template, obligatoire pour ITO
my $NomTemplate=shift(@ARGV);

my $RetourITO = LectureParametres();
$RetourITO = Traitement() if($RetourITO == 0);

Trace "$NomTemplate=$RetourITO" if($debug);

# envoi du code retour a ITO via opcmon
if(!$NomTemplate || $NomTemplate eq "bidon") {
	exit $RetourITO;
} else {
	opcmon("$NomTemplate=$RetourITO");
	exit 0;
}

__END__
